package com.atlassian.bitbucket.repository;

import javax.annotation.Nonnull;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * A simple, immutable implementation of {@link MinimalRefChange}.
 *
 * @since 7.11
 */
public class SimpleMinimalRefChange implements MinimalRefChange {

    private final MinimalRef ref;
    private final RefChangeType type;

    protected SimpleMinimalRefChange(AbstractMinimalRefChangeBuilder<?, ?> builder) {
        ref = requireNonNull(builder.ref, "ref");
        type = requireNonNull(builder.type, "type");
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SimpleMinimalRefChange that = (SimpleMinimalRefChange) o;
        return Objects.equals(ref.getId(), that.ref.getId()) &&
               Objects.equals(ref.getType(), that.ref.getType()) &&
               type == that.type;
    }

    @Nonnull
    @Override
    public MinimalRef getRef() {
        return ref;
    }

    @Nonnull
    @Override
    public RefChangeType getType() {
        return type;
    }

    @Override
    public int hashCode() {
        return Objects.hash(ref.getId(), ref.getType(), type);
    }

    @Override
    public String toString() {
        return getRef().getId() + " (" + getType() + ")";
    }

    public static class Builder extends AbstractMinimalRefChangeBuilder<Builder, SimpleMinimalRefChange> {

        @Nonnull
        @Override
        public SimpleMinimalRefChange build() {
            return new SimpleMinimalRefChange(this);
        }

        @Override
        protected Builder self() {
            return this;
        }
    }

    protected abstract static class AbstractMinimalRefChangeBuilder<B extends AbstractMinimalRefChangeBuilder<B, R>, R extends MinimalRefChange> {

        private MinimalRef ref;
        private RefChangeType type;

        public AbstractMinimalRefChangeBuilder() {
        }

        public AbstractMinimalRefChangeBuilder(@Nonnull R refChange) {
            requireNonNull(refChange, "refChange");

            ref = refChange.getRef();
            type = refChange.getType();
        }

        @Nonnull
        public abstract R build();

        @Nonnull
        public B ref(@Nonnull MinimalRef value) {
            ref = requireNonNull(value, "value");

            return self();
        }

        @Nonnull
        public B type(@Nonnull RefChangeType value) {
            type = requireNonNull(value, "value");

            return self();
        }

        protected abstract B self();
    }
}
