package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

import static java.util.Optional.ofNullable;

/**
 * Request for searching pull request participant users.
 *
 * @since 4.2
 */
public class PullRequestParticipantSearchRequest {

    private final String filter;
    private final Repository fromRepository;
    private final PullRequestRole role;
    private final Repository toRepository;

    private PullRequestParticipantSearchRequest(Builder builder) {
        fromRepository = builder.fromRepository;
        role = builder.role;
        filter = builder.filter;
        toRepository = builder.toRepository;
    }

    /**
     * When set, limits the search to the users whose username, name or email address contain the {@code filter} value.
     *
     * @return the filter
     */
    @Nonnull
    public Optional<String> getFilter() {
        return ofNullable(filter);
    }

    /**
     * When set, limits the search by the repository the pull requests are coming from.
     *
     * @return the from repository
     */
    @Nonnull
    public Optional<Repository> getFromRepository() {
        return ofNullable(fromRepository);
    }

    /**
     * When set, limits the search by the participants role.
     *
     * @return the participant role
     */
    @Nonnull
    public Optional<PullRequestRole> getRole() {
        return ofNullable(role);
    }

    /**
     * When set, limits the search by the repository the pull requests are going to.
     *
     * @return the to repository
     */
    @Nonnull
    public Optional<Repository> getToRepository() {
        return ofNullable(toRepository);
    }

    public static class Builder {

        private String filter;
        private Repository fromRepository;
        private PullRequestRole role;
        private Repository toRepository;

        /**
         * Assembles a new {@link PullRequestParticipantSearchRequest} from the provided values.
         *
         * @return a new pull request participant search request instance
         */
        @Nonnull
        public PullRequestParticipantSearchRequest build() {
            return new PullRequestParticipantSearchRequest(this);
        }

        /**
         * Limits the search to the users whose username, name or email address contain the {@code value}
         *
         * @param value the filter string
         * @return the current builder
         */
        @Nonnull
        public Builder filter(@Nullable String value) {
            filter = value;
            return this;
        }

        /**
         * Limits the search by the repository the pull requests are coming from.
         *
         * @param value the repository to restrict by
         * @return the current builder
         */
        @Nonnull
        public Builder fromRepository(@Nullable Repository value) {
            fromRepository = value;
            return this;
        }

        /**
         * Limits the search by the participants role.
         *
         * @param value the role to restrict by
         * @return the current builder
         */
        @Nonnull
        public Builder role(@Nullable PullRequestRole value) {
            role = value;
            return this;
        }

        /**
         * Limits the search by the repository the pull requests are going to.
         *
         * @param value the repository to restrict by
         * @return the current builder
         */
        @Nonnull
        public Builder toRepository(@Nullable Repository value) {
            toRepository = value;
            return this;
        }
    }
}
