package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.bitbucket.validation.annotation.OptionalString;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Describes a participant on a pull request.
 * <p>
 * Each participant has a {@link PullRequestRole role}, such as {@link PullRequestRole#AUTHOR author} or
 * {@link PullRequestRole#REVIEWER reviewer}.
 *
 * @see PullRequest#getAuthor()
 * @see PullRequest#getParticipants()
 * @see PullRequest#getReviewers()
 */
public interface PullRequestParticipant extends Comparable<PullRequestParticipant> {

    /**
     * The commit hash at which this participant last reviewed the pull request.
     * <p>
     * A participant is considered to have reviewed a pull request when their status is changed to
     * {@link PullRequestParticipantStatus#NEEDS_WORK} or {@link PullRequestParticipantStatus#APPROVED}. A pull request
     * being rescoped does not trigger updating of participant's last reviewed commits so it is possible for the commit
     * hash returned to no longer be part of the {@link #getPullRequest() pull request}.
     * <p>
     * This method will return {@code null} if the participant has not yet reviewed the pull request. The last
     * reviewed commit for a participant may also be reset back to {@code null} if changes to the pull request mean the
     * commit hash may no longer be suitable for the application to use internally.
     *
     * @return the commit hash at which this participant last reviewed the pull request
     * @since 4.9
     */
    @Nullable
    @OptionalString(minimumSize = 40, size = 40)
    String getLastReviewedCommit();

    /**
     * @return the pull request this participant is associated with
     */
    @Nonnull
    PullRequest getPullRequest();

    /**
     * @return the role of this participant in the pull request.
     */
    @Nonnull
    PullRequestRole getRole();

    /**
     * @return the status of this participant in the pull request.
     * @since 4.2
     */
    @Nonnull
    PullRequestParticipantStatus getStatus();

    /**
     * @return the user participating in the pull request
     */
    @Nonnull
    ApplicationUser getUser();

    /**
     * @return {@code true} if the status of this participant is {@link PullRequestParticipantStatus#APPROVED APPROVED}
     * {@code false} otherwise
     */
    boolean isApproved();

    /**
     * Sorts participants first by status, then by name
     *
     * @param other the participant to compare to
     * @return <ol>
     * <li>{@code 0} if {@code other} has same status and same name</li>
     * <li>{@code < 0} if {@code other} has approved the pull request and {@code this} has not, or if {@code other} has
     * same status but comes first when alphabetically ordered by name</li>
     * <li>{@code > 0} otherwise</li>
     * </ol>
     * @throws NullPointerException if {@code other} is {@code null}
     * @since 4.2
     */
    @Override
    int compareTo(@Nonnull PullRequestParticipant other);
}
