package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.EntityOutOfDateException;
import com.atlassian.bitbucket.i18n.KeyedMessage;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * A specialization of {@link EntityOutOfDateException} thrown when a {@link PullRequest}'s
 * {@link PullRequest#getVersion() version} is out-of-date.
 */
public class PullRequestOutOfDateException extends EntityOutOfDateException {

    private static final long serialVersionUID = 2;

    private final PullRequest pullRequest;

    /**
     * Constructs a {@code PullRequestOutOfDateException} with {@link #UNKNOWN_VERSION unspecified} versions.
     * Because this constructor provides so little information, it should be avoided except in situaitons where
     * there is simply no additional information available.
     *
     * @param message a message describing the error
     * @param cause   the triggering exception, or {@code null} if no cause is available
     */
    public PullRequestOutOfDateException(@Nonnull KeyedMessage message, @Nullable Throwable cause) {
        super(message, cause, UNKNOWN_VERSION, UNKNOWN_VERSION);

        pullRequest = null;
    }

    /**
     * Constructs a {@code PullRequestOutOfDateException}, providing the <i>current</i> {@link PullRequest} and
     * the version that was {@link #getExpectedVersion() expected}.
     *
     * @param message         a message describing the error
     * @param pullRequest     the <i>current</i> pull request state
     * @param expectedVersion the expected pull request version, generally older (lower) than the current version
     */
    public PullRequestOutOfDateException(@Nonnull KeyedMessage message, @Nonnull PullRequest pullRequest,
                                         int expectedVersion) {
        super(message, expectedVersion, requireNonNull(pullRequest, "pullRequest").getVersion());

        this.pullRequest = pullRequest;
    }

    /**
     * Constructs a {@code PullRequestOutOfDateException}, providing the <i>current</i> {@link PullRequest} and
     * the version that was {@link #getExpectedVersion() expected}.
     *
     * @param message         a message describing the error
     * @param cause           the triggering exception, or {@code null} if no cause is available
     * @param pullRequest     the <i>current</i> pull request state
     * @param expectedVersion the expected pull request version, generally older (lower) than the current version
     */
    public PullRequestOutOfDateException(@Nonnull KeyedMessage message, @Nullable Throwable cause,
                                         @Nonnull PullRequest pullRequest, int expectedVersion) {
        super(message, cause, expectedVersion, requireNonNull(pullRequest, "pullRequest").getVersion());

        this.pullRequest = pullRequest;
    }

    /**
     * Retrieves the <i>current</i> state for the pull request, if it was available when the exception was thrown.
     * When available, the {@link PullRequest#getVersion() version} of the returned pull request will always match
     * the {@link #getCurrentVersion() current version}.
     *
     * @return the <i>current</i> pull request, or {@code null} if the current state was not available
     */
    @Nullable
    public PullRequest getPullRequest() {
        return pullRequest;
    }
}
