package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.comment.Comment;
import com.atlassian.bitbucket.comment.CommentAction;
import com.atlassian.bitbucket.event.pull.*;

/**
 * Enumerates the actions which can be taken on a pull request, defining the {@link PullRequestActivity activities}
 * that may occur. Each action may have an associated {@link PullRequestActivity activity} type.
 */
public enum PullRequestAction {

    /**
     * A user has approved a {@link PullRequest pull request}
     * <ul>
     *     <li>Activity: {@link PullRequestActivity}</li>
     *     <li>Event: {@link PullRequestParticipantApprovedEvent PullRequestParticipantApprovedEvent}</li>
     * </ul>
     */
    APPROVED(7),
    /**
     * A comment has been {@link CommentAction added, deleted, edited or replied to} on
     * the {@link PullRequest pull request}.
     * <ul>
     *     <li>Activity: {@link PullRequestCommentActivity}</li>
     *     <li>Events:
     *     <ul>
     *         <li>{@link PullRequestCommentAddedEvent PullRequestCommentAddedEvent}</li>
     *         <li>{@link PullRequestCommentDeletedEvent PullRequestCommentDeletedEvent}</li>
     *         <li>{@link PullRequestCommentEditedEvent PullRequestCommentEditedEvent}</li>
     *         <li>{@link PullRequestCommentRepliedEvent PullRequestCommentRepliedEvent}</li>
     *     </ul>
     *     </li>
     * </ul>
     */
    COMMENTED(1),
    /**
     * The {@link PullRequest pull request} has been rejected and declined without merging.
     * <ul>
     *     <li>Activity: {@link PullRequestActivity}</li>
     *     <li>Event: {@link PullRequestDeclinedEvent PullRequestDeclinedEvent}</li>
     * </ul>
     */
    DECLINED(0),
    /**
     * The {@link PullRequest pull request} has been deleted.
     * <ul>
     *     <li>Event: {@link PullRequestDeletedEvent PullRequestDeletedEvent}</li>
     * </ul>
     * <p>
     * <b>Note</b>: Since the pull request has been deleted, along with all of its activities and comments, this
     * action will never be returned by an {@link PullRequestActivity#getAction activity}.
     *
     * @since 5.1
     */
    DELETED(10),
    /**
     * The {@link PullRequest pull request} has been accepted and merged to the {@link PullRequest#getToRef() target}.
     * <ul>
     *     <li>Activity: {@link PullRequestMergeActivity}</li>
     *     <li>Event: {@link PullRequestMergedEvent PullRequestMergedEvent}</li>
     * </ul>
     */
    MERGED(2),
    /**
     * A new {@link PullRequest pull request} has been opened.
     * <ul>
     *     <li>Activity: {@link PullRequestActivity}</li>
     *     <li>Events:
     *     <ul>
     *         <li>{@link PullRequestOpenRequestedEvent PullRequestOpenRequestedEvent}</li>
     *         <li>{@link PullRequestOpenedEvent PullRequestOpenedEvent}</li>
     *     </ul>
     *     </li>
     * </ul>
     */
    OPENED(3),
    /**
     * A previously declined {@link PullRequest pull request} has been reopened.
     * <ul>
     *     <li>Activity: {@link PullRequestActivity}</li>
     *     <li>Event: {@link PullRequestReopenedEvent PullRequestReopenedEvent}</li>
     * </ul>
     */
    REOPENED(4),
    /**
     * The scope of the {@link PullRequest pull request} has been changed and commits have been added or removed.
     * <ul>
     *     <li>Activity: {@link PullRequestRescopeActivity}</li>
     *     <li>Event: {@link PullRequestRescopedEvent PullRequestRescopedEvent}</li>
     * </ul>
     */
    RESCOPED(5),
    /**
     * A {@link Comment pending comment} has been {@link CommentAction added or replied} on the
     * {@link PullRequest pull request}.
     * NOTE: This action does not create an {@link PullRequestActivity activity} item.
     * <ul>
     *     <li>Event:
     *     <ul>
     *         <li>{@link PullRequestReviewCommentAddedEvent PullRequestReviewCommentAddedEvent}</li>
     *         <li>{@link PullRequestReviewCommentRepliedEvent PullRequestReviewCommentRepliedEvent}</li>
     *     </ul>
     *     </li>
     * </ul>
     * @since 7.7
     */
    REVIEW_COMMENTED(11),
    /**
     * Discard a user's pull request review where {@link Comment pending comments} were added to a
     * {@link PullRequest pull request}.
     * NOTE: This action does not create an {@link PullRequestActivity activity} item.
     * <ul>
     *     <li>Event: {@link PullRequestReviewDiscardedEvent PullRequestReviewDiscardedEvent} </li>
     * </ul>
     * @since 7.7
     */
    REVIEW_DISCARDED(12),
    /**
     * The pull request review where the user has added {@link Comment pending comments} on a
     * {@link PullRequest pull request} is finished.
     * NOTE: This action does not create an {@link PullRequestActivity activity} item.
     * <ul>
     *     <li>Event: {@link PullRequestReviewFinishedEvent PullRequestReviewFinishedEvent} </li>
     * </ul>
     * @since 7.7
     */
    REVIEW_FINISHED(13),
    /**
     * A user has finished reviewing the changes in a {@link PullRequest pull request}.
     * They have provided feedback which needs to be addressed before they can approve.
     * NOTE: This action does not create an {@link PullRequestActivity activity} item.
     * <ul>
     *     <li>Event: {@link PullRequestParticipantReviewedEvent PullRequestParticipantReviewedEvent}</li>
     * </ul>
     * @since 4.4
     */
    REVIEWED(9),
    /**
     * A user has removed a approval from a {@link PullRequest pull request}
     * <ul>
     *     <li>Activity: {@link PullRequestActivity}</li>
     *     <li>Event: {@link PullRequestParticipantUnapprovedEvent PullRequestParticipantUnapprovedEvent}</li>
     * </ul>
     */
    UNAPPROVED(8),
    /**
     * The details of the {@link PullRequest pull request}, such as title or description, have been updated.
     * <ul>
     *     <li>Activity: {@link PullRequestActivity}</li>
     *     <li>Event: {@link PullRequestUpdatedEvent PullRequestUpdatedEvent}</li>
     * </ul>
     */
    UPDATED(6);

    private final int id;

    PullRequestAction(int id) {
        this.id = id;
    }

    public static PullRequestAction fromId(int id) {
        for (PullRequestAction value : values()) {
            if (value.getId() == id) {
                return value;
            }
        }
        throw new IllegalArgumentException("No PullRequestAction is associated with ID [" + id + "]");
    }

    public int getId() {
        return id;
    }
}
