package com.atlassian.bitbucket.property;

import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import java.util.Map;

import static java.util.Objects.requireNonNull;

/**
 * An implementation of {@link PropertySupport} for use in SCM entities, with write-once immutability provided by
 * {@link #setProperties(PropertyMap)}. Note: The recommended way to use this class is by composition.
 */
public class SimplePropertySupport implements PropertySupport {

    private transient PropertyMap properties;

    public SimplePropertySupport(@Nonnull AbstractPropertyBuilder<?> builder) {
        properties = builder.properties.build();
    }

    @Nonnull
    @Override
    public PropertyMap getProperties() {
        return properties;
    }

    /**
     * Set properties on this entity.
     * If the properties on this instance are already set, this will merge the {@code properties}.
     * The new properties will overwrite existing properties with the same key.
     *
     * @param properties the properties to set
     * @see #getProperties() to query the current state of the properties
     */
    public void setProperties(@Nonnull PropertyMap properties) {
        requireNonNull(properties, "properties");

        PropertyMap.Builder builder = new PropertyMap.Builder();
        if (!(this.properties == null || this.properties.isEmpty())) {
            //Apply any existing properties first so new ones can "override" existing ones
            builder.properties(this.properties);
        }

        this.properties = builder.properties(properties).build();
    }

    public abstract static class AbstractPropertyBuilder<B extends AbstractPropertyBuilder<B>> extends BuilderSupport {

        protected final PropertyMap.Builder properties = new PropertyMap.Builder();

        protected AbstractPropertyBuilder() {
        }
        
        protected AbstractPropertyBuilder(@Nonnull PropertySupport entity) {
            properties(requireNonNull(entity, "entity").getProperties());
        }

        @Nonnull
        public B property(@Nonnull String key, @Nonnull Object value) {
            properties.property(key, value);

            return self();
        }

        @Nonnull
        public B properties(@Nonnull Map<String, Object> other) {
            properties.properties(other);

            return self();
        }

        @Nonnull
        protected abstract B self();
    }
}
