package com.atlassian.bitbucket.project;

import com.atlassian.bitbucket.util.BuilderSupport;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;
import static com.google.common.base.Preconditions.checkState;

public abstract class AbstractProjectRequest {

    private final String description;
    private final String key;
    private final String name;
    private final boolean publiclyAccessible;

    /**
     * @param builder the builder
     * @since 4.2
     */
    AbstractProjectRequest(AbstractBuilder builder) {
        checkState(StringUtils.isNotBlank(builder.key), "The key of the the project is required");
        checkState(StringUtils.isNotBlank(builder.name), "The name of the project is required");

        description = builder.description;
        name = builder.name;
        key = builder.key;
        publiclyAccessible = builder.publiclyAccessible;
    }

    /**
     * @return an optional description for the project
     */
    @Nullable
    public String getDescription() {
        return description;
    }

    /**
     * @return the unique key to identify the project, always stored in uppercase
     */
    @Nonnull
    public String getKey() {
        return key;
    }

    /**
     * @return the unique name for the project
     */
    @Nonnull
    public String getName() {
        return name;
    }

    /**
     * @return {@code true} if the project will be made publicly accessible, {@code false} otherwise.
     */
    public boolean isPublic() {
        return publiclyAccessible;
    }

    public abstract static class AbstractBuilder<B extends AbstractBuilder> extends BuilderSupport {

        private String description;
        private String key;
        private String name;
        private boolean publiclyAccessible;

        protected AbstractBuilder() {
        }

        protected AbstractBuilder(@Nonnull Project project) {
            description = requireNonNull(project, "project").getDescription();
            key = project.getKey();
            name = project.getName();
            publiclyAccessible = project.isPublic();
        }

        @Nonnull
        public B description(@Nullable String value) {
            description = value;

            return self();
        }

        @Nonnull
        public B key(@Nonnull String value) {
            key = requireNonNull(value, "key");

            return self();
        }

        @Nonnull
        public B name(@Nonnull String value) {
            name = requireNonNull(value, "name");

            return self();
        }

        /**
         * @param value whether the repository will be publicly accessible or not.
         * @return {@code this}
         */
        @Nonnull
        public B publiclyAccessible(boolean value) {
            publiclyAccessible = value;

            return self();
        }

        @Nonnull
        protected abstract B self();
    }
}
