package com.atlassian.bitbucket.migration.event;

import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.bitbucket.job.Job;
import com.atlassian.bitbucket.job.JobState;
import com.atlassian.event.api.AsynchronousPreferred;

import javax.annotation.Nonnull;

/**
 * Event that is raised when an Data Center export has completed executing.
 * <p>
 * To determine whether the export has finished successfully, the {@link #getFinalState() state} should be inspected.
 *
 * @since 5.13
 */
@AsynchronousPreferred
@EventName("stash.migration.export.finished")
public class MigrationExportFinishedEvent extends AbstractInstanceMigrationEvent {

    private final long errorCount;
    private final long infoCount;
    private final long warningCount;

    public MigrationExportFinishedEvent(@Nonnull Object source, @Nonnull Job job, long errorCount,
                                        long infoCount, long warningCount) {
        super(source, job);
        this.errorCount = errorCount;
        this.infoCount = infoCount;
        this.warningCount = warningCount;
    }

    /**
     * @return count of export error messages
     */
    public long getErrorCount() {
        return errorCount;
    }

    /**
     * @return The {@link JobState#getId()} that the export job was in when it completed
     */
    public int getFinalState() {
        return job.getState().getId();
    }

    /**
     * @return count of export info messages
     */
    public long getInfoCount() {
        return infoCount;
    }

    /**
     * @return count of export info messages
     */
    public long getWarningCount() {
        return warningCount;
    }
}
