package com.atlassian.bitbucket.mesh;

import javax.annotation.Nonnull;
import java.util.List;
import java.util.Optional;

/**
 * A service for administering {@link MeshNode}s.
 * <p>
 * Mesh is a distributed Git management system, allowing repositories to replicated to multiple <i>active</i> nodes,
 * all of which can serve both read and write traffic. Adding Mesh nodes makes additional targets available to host
 * replicas and increases the overall availability and scalability of the system.
 *
 * @since 8.0
 */
public interface MeshService {

    /**
     * @param id the node ID
     * @return the node
     * @throws NoSuchMeshNodeException if the node is not found
     */
    @Nonnull
    MeshNode getMember(long id);

    /**
     * @return the list of currently registered nodes, including the sidecar if it's running
     */
    @Nonnull
    List<MeshNode> getMembers();

    /**
     * @return the sidecar, if one is running on the current node
     */
    @Nonnull
    Optional<MeshNode> getSidecar();

    /**
     * @return {@code true} if the Mesh feature is available to use; {@code false} otherwise. The Mesh feature is
     *         only available if a data center license is installed.
     */
    boolean isAvailable();

    /**
     * Registers and provisions a new node and assigns unallocated partitions to it. The node is <i>required</i> to
     * be running so it can be provisioned, which establishes trust between the control plane and the node.
     *
     * @param request request providing the new node details such as the name, id, RPC URL and secret
     * @return the newly registered node, with a generated value for {@link MeshNode#getId() ID} if one was not
     *         provided in the {@code request}
     */
    @Nonnull
    MeshNode register(@Nonnull RegisterMeshNodeRequest request);

    /**
     * Updates the node details, such as the RPC URL, name and/or secret.
     *
     * @param request provides new values for fields that should be updated
     * @return the updated node
     * @throws NoSuchMeshNodeException if the node is not found
     */
    @Nonnull
    MeshNode update(@Nonnull UpdateMeshNodeRequest request);
}
