package com.atlassian.bitbucket.job;

import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;
import java.util.OptionalInt;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.trimToNull;

public class JobUpdateRequest {

    private final Job job;
    private final Integer progressPercentage;
    private final String progressMessage;
    private final JobState state;

    private JobUpdateRequest(Builder builder) {
        job = builder.job;
        progressPercentage = builder.progressPercentage;
        progressMessage = builder.progressMessage;
        state = builder.state;
    }

    @Nonnull
    public Job getJob() {
        return job;
    }

    @Nonnull
    public OptionalInt getProgressPercentage() {
        return progressPercentage == null ? OptionalInt.empty() : OptionalInt.of(progressPercentage);
    }

    @Nonnull
    public Optional<String> getProgressMessage() {
        return Optional.ofNullable(progressMessage);
    }

    @Nonnull
    public Optional<JobState> getState() {
        return Optional.ofNullable(state);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        JobUpdateRequest that = (JobUpdateRequest) o;
        return Objects.equal(getJob().getId(), that.getJob().getId()) &&
                Objects.equal(getProgressPercentage(), that.getProgressPercentage()) &&
                Objects.equal(getProgressMessage(), that.getProgressMessage()) &&
                Objects.equal(getState(), that.getState());
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(getJob(), getProgressPercentage(), getProgressMessage(), getState());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("job.id", job.getId())
                .add("progressPercentage", progressPercentage)
                .add("progressMessage", progressMessage)
                .add("state", state)
                .toString();
    }

    public static class Builder {

        private final Job job;

        private String progressMessage;
        private Integer progressPercentage;
        private JobState state;

        public Builder(@Nonnull Job job) {
            this.job = requireNonNull(job, "job");
        }

        @Nonnull
        public JobUpdateRequest build() {
            return new JobUpdateRequest(this);
        }

        @Nonnull
        public Builder progressMessage(@Nullable String value) {
            progressMessage = trimToNull(value);
            return this;
        }

        @Nonnull
        public Builder progressPercentage(int value) {
            progressPercentage = Math.max(0, Math.min(value, 100));
            return this;
        }

        @Nonnull
        public Builder state(@Nullable JobState value) {
            state = value;
            return this;
        }
    }
}