package com.atlassian.bitbucket.job;

import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.trimToNull;

/**
 * A request to create a {@link JobMessage message} for a {@link Job job}
 *
 * @since 5.13
 */
public class JobMessageCreationRequest {

    private final Job job;
    private final JobMessageSeverity severity;
    private final String subject;
    private final String text;

    private JobMessageCreationRequest(Builder builder) {
        job = requireNonNull(builder.job, "job");
        severity = builder.severity;
        subject = builder.subject;
        text = requireNonNull(builder.text, "text");
    }

    @Nonnull
    public Job getJob() {
        return job;
    }

    @Nonnull
    public JobMessageSeverity getSeverity() {
        return severity;
    }

    @Nonnull
    public Optional<String> getSubject() {
        return ofNullable(subject);
    }

    @Nonnull
    public String getText() {
        return text;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        JobMessageCreationRequest that = (JobMessageCreationRequest) o;
        return Objects.equal(getJob().getId(), that.getJob().getId()) &&
                Objects.equal(getSeverity(), that.getSeverity()) &&
                Objects.equal(getSubject(), that.getSubject()) &&
                Objects.equal(getText(), that.getText());
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(getJob(), getSubject(), getSeverity(), getText());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("job.id", job.getId())
                .add("severity", severity)
                .add("subject", subject)
                .add("text", text)
                .toString();
    }
    
    public static class Builder {

        public Job job;
        public JobMessageSeverity severity;
        public String subject;
        public String text;

        public Builder(@Nonnull Job job, @Nonnull String text) {
            this.job = requireNonNull(job, "job");
            this.text = requireNonNull(text, "text");

            severity = JobMessageSeverity.INFO;
        }

        @Nonnull
        public Builder severity(@Nonnull JobMessageSeverity value) {
            severity = requireNonNull(value, "severity");
            return this;
        }

        @Nonnull
        public Builder subject(@Nullable String value) {
            subject = trimToNull(value);
            return this;
        }

        @Nonnull
        public JobMessageCreationRequest build() {
            return new JobMessageCreationRequest(this);
        }
    }
}