package com.atlassian.bitbucket.hook.script;

import com.atlassian.bitbucket.io.InputSupplier;
import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.trimToNull;

/**
 * @since 6.2
 */
public class HookScriptUpdateRequest {

    private final InputSupplier<InputStream> content;
    private final String description;
    private final String name;
    private final String pluginKey;
    private final HookScript script;

    private HookScriptUpdateRequest(Builder builder) {
        content = builder.content;
        description = builder.description;
        name = builder.name;
        pluginKey = builder.pluginKey;
        script = builder.script;
    }

    @Nonnull
    public Optional<InputSupplier<InputStream>> getContent() {
        return ofNullable(content);
    }

    @Nonnull
    public Optional<String> getDescription() {
        return ofNullable(description);
    }

    @Nonnull
    public String getName() {
        return name;
    }

    @Nonnull
    public String getPluginKey() {
        return pluginKey;
    }

    @Nonnull
    public HookScript getScript() {
        return script;
    }

    public static class Builder extends BuilderSupport {

        private final HookScript script;

        private InputSupplier<InputStream> content;
        private String description;
        private String name;
        private String pluginKey;

        public Builder(@Nonnull HookScript script) {
            this.script = requireNonNull(script, "script");

            description = script.getDescription().orElse(null);
            name = script.getName();
            pluginKey = script.getPluginKey();
        }

        @Nonnull
        public HookScriptUpdateRequest build() {
            return new HookScriptUpdateRequest(this);
        }

        @Nonnull
        public Builder content(@Nonnull InputSupplier<InputStream> value) {
            content = requireNonNull(value, "content");

            return this;
        }

        @Nonnull
        public Builder content(@Nonnull String value) {
            //Hold onto the bytes, rather than the String
            return content(requireNonBlank(value, "content").getBytes(StandardCharsets.UTF_8));
        }

        @Nonnull
        public Builder content(@Nonnull byte[] value) {
            if (requireNonNull(value, "content").length == 0) {
                throw new IllegalArgumentException("Hook script contents cannot be empty");
            }
            content = () -> new ByteArrayInputStream(value);

            return this;
        }

        @Nonnull
        public Builder description(@Nullable String value) {
            description = trimToNull(value);

            return this;
        }

        @Nonnull
        public Builder name(@Nonnull String value) {
            name = requireNonBlank(value, "name");

            return this;
        }

        @Nonnull
        public Builder pluginKey(@Nonnull String value) {
            pluginKey = requireNonBlank(value, "pluginKey");

            return this;
        }
    }
}
