package com.atlassian.bitbucket.hook.repository;

import com.atlassian.bitbucket.avatar.CacheableAvatarSupplier;
import com.atlassian.bitbucket.scope.Scope;
import com.atlassian.bitbucket.setting.Settings;
import com.atlassian.bitbucket.setting.SettingsBuilder;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;
import com.atlassian.bitbucket.validation.ArgumentValidationException;
import com.atlassian.bitbucket.validation.FormValidationException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Handles the aggregation of hooks and their enabled status.
 * Settings for these hooks are handled elsewhere.
 */
public interface RepositoryHookService {

    /**
     * Create a new {@link SettingsBuilder} which can be used to create a {@link Settings} instance.
     *
     * @return a new builder
     */
    @Nonnull
    SettingsBuilder createSettingsBuilder();

    /**
     * Delete the hook configuration specified in the request.
     *
     * @param request describes the hook for which the configuration should be deleted
     * @since 5.2
     * @throws ArgumentValidationException if the provided request is for any {@link Scope} other than
     *         {@link com.atlassian.bitbucket.scope.ScopeType#REPOSITORY REPOSITORY}
     */
    void delete(@Nonnull DeleteRepositoryHookRequest request);

    /**
     * Disable the hook specified in the request.
     *
     * @param request describing the hook to be disabled
     * @return the newly updated {@link RepositoryHook}
     * @since 5.2
     */
    @Nonnull
    RepositoryHook disable(@Nonnull DisableRepositoryHookRequest request);

    /**
     * Enable and apply settings for the hook specified in the request.
     *
     * @param request details for enabling a repository hook
     * @return the newly updated {@link RepositoryHook}
     * @throws FormValidationException     when settings are invalid, leaving the existing version unchanged
     * @throws ArgumentValidationException if the settings to be saved are too large, the current limit is 32KB
     *                                     once serialized.
     * @since 5.2
     */
    @Nonnull
    RepositoryHook enable(@Nonnull EnableRepositoryHookRequest request) throws FormValidationException;

    /**
     * Returns a supplier of an avatar for this hook, either one specified in the module or a default.
     *
     * @param hookKey the hook key with the desired avatar
     * @return an supplier for streaming the avatar
     */
    @Nonnull
    CacheableAvatarSupplier getAvatar(@Nonnull String hookKey);

    /**
     * Find an effective installed hook with the specified key enabled status for this scope.<br>
     * <p>
     * An effective hook is a hook that is either:
     * <ul>
     * <li>configured at the given scope,</li>
     * <li>or configured at a higher scope.</li>
     * </ul>
     * {@link RepositoryHook Repository hooks} can be configured at
     * {@link com.atlassian.bitbucket.scope.ProjectScope project scope} or
     * {@link com.atlassian.bitbucket.scope.RepositoryScope repository scope}
     *
     * @param scope   the scope to load the hook from
     * @param hookKey the hook to lookup
     * @return the effective {@link RepositoryHook} if found. Otherwise {@code null}.
     * @since 5.2
     */
    @Nullable
    RepositoryHook getByKey(@Nonnull Scope scope, @Nonnull String hookKey);

    /**
     * Gets the effective settings for the hook specified in the request.
     *
     * @param request details to get the settings for the specified hook
     * @return the effective settings currently stored or {@code null} if no settings are stored.
     * @since 5.2
     */
    @Nullable
    RepositoryHookSettings getSettings(@Nonnull GetRepositoryHookSettingsRequest request);

    /**
     * Migrates settings between the specified hook keys.
     * <p>
     * Hook keys are based on the combination of the plugin's key and the {@code <repository-hook/>}'s {@code key}
     * attribute. Changing <i>either key</i> will result in a new hook key, and the "loss" of all existing settings
     * in all repositories (because they remain associated with the old key).
     * <p>
     * When calling this method, the {@code <repository-hook/>} for the specified {@code oldHookKey} must no longer
     * be installed, and the {@code <repository-hook/>} for the new key must be present.
     * <p>
     * Calling this method requires {@link com.atlassian.bitbucket.permission.Permission#ADMIN ADMIN} permission.
     * Apps may need to use the {@link com.atlassian.bitbucket.user.SecurityService#withPermission SecurityService}
     * to apply that permission.
     *
     * @param oldHookKey the hook's old key
     * @param newHookKey the hook's new key
     * @return the number of projects/repositories for which hook settings were migrated
     * @since 5.14
     * @throws ArgumentValidationException if the old hook is still installed, or the new hook is not installed
     */
    int migrateSettings(@Nonnull String oldHookKey, @Nonnull String newHookKey);

    /**
     * Calls all registered and enabled {@code repository-hook} modules ({@code PostRepositoryHook})
     *
     * @param request describes the ref changes that have been applied
     * @param <T> type of the request
     * @since 5.0
     */
    <T extends RepositoryHookRequest> void postUpdate(@Nonnull T request);

    /**
     * Calls all registered and enabled {@code repository-hook} modules ({@code PreRepositoryHook})
     *
     * @param request describes the ref changes that are about to be made
     * @param <T> type of the request
     * @return the outcome of the {@code repository-hook} modules
     * @since 5.0
     */
    @Nonnull
    <T extends RepositoryHookRequest> RepositoryHookResult preUpdate(@Nonnull T request);

    /**
     * Searches for {@link RepositoryHook repository hooks} that match the provided {@link RepositoryHookSearchRequest request}.
     *
     * @param request     a request object describing the repository hooks to return
     * @param pageRequest paging options
     * @return a page of {@link RepositoryHook}s matching the supplied type
     * @since 5.2
     */
    @Nonnull
    Page<RepositoryHook> search(@Nonnull RepositoryHookSearchRequest request, @Nonnull PageRequest pageRequest);

    /**
     * Sets the settings for the hook specified in the request.
     *
     * @param request describing the update settings request
     * @return the saved settings.
     * @throws FormValidationException     when settings are invalid, leaving the existing version unchanged
     * @throws ArgumentValidationException if the settings to be saved are too large, the current limit is 32KB
     *                                     once serialized.
     * @see #createSettingsBuilder()
     * @since 5.2
     */
    @Nonnull
    Settings setSettings(@Nonnull SetRepositoryHookSettingsRequest request) throws FormValidationException;
}
