package com.atlassian.bitbucket.hook.repository;

import com.atlassian.bitbucket.hook.ScmHookDetails;
import com.atlassian.bitbucket.repository.RefChange;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;

/**
 * Base class for the various {@link RepositoryHookRequest RepositoryHookRequests}. Plugins that want to invoke
 * {@link RepositoryHookService#preUpdate(RepositoryHookRequest) pre-hooks} or
 * {@link RepositoryHookService#postUpdate(RepositoryHookRequest) post-hooks} with a custom request type can use
 * this class as a base class to provide the common request attributes.
 *
 * @since 5.0
 */
public abstract class AbstractRepositoryHookRequest implements RepositoryHookRequest {

    private final Map<String, Object> context;
    private final boolean dryRun;
    private final Repository repository;
    private final RepositoryHookTrigger trigger;

    protected AbstractRepositoryHookRequest(AbstractBuilder<?> builder) {
        context = builder.context.build();
        dryRun = builder.dryRun;
        repository = builder.repository;
        trigger = builder.trigger;
    }

    @Nonnull
    @Override
    public Map<String, Object> getContext() {
        return context;
    }

    @Nonnull
    @Override
    public Collection<RefChange> getRefChanges() {
        return Collections.emptyList();
    }

    @Nonnull
    @Override
    public Repository getRepository() {
        return repository;
    }

    @Nonnull
    @Override
    public Optional<ScmHookDetails> getScmHookDetails() {
        return empty();
    }

    @Nonnull
    @Override
    public RepositoryHookTrigger getTrigger() {
        return trigger;
    }

    @Override
    public boolean isDryRun() {
        return dryRun;
    }

    protected abstract static class AbstractBuilder<B extends AbstractBuilder<B>> extends BuilderSupport {

        private final ImmutableMap.Builder<String, Object> context;
        private final Repository repository;
        private final RepositoryHookTrigger trigger;

        private boolean dryRun;

        protected AbstractBuilder(@Nonnull Repository repository, @Nonnull RepositoryHookTrigger trigger) {
            this.repository = requireNonNull(repository, "repository");
            this.trigger = requireNonNull(trigger, "trigger");

            context = ImmutableMap.builder();
        }

        @Nonnull
        public B context(@Nonnull String key, @Nonnull String value) {
            context.put(requireNonNull(key, "key"), requireNonNull(value, "value"));

            return self();
        }

        @Nonnull
        public B context(@Nonnull Map<String, Object> value) {
            context.putAll(requireNonNull(value, "context"));

            return self();
        }

        @Nonnull
        public B dryRun(boolean value) {
            dryRun = value;
            return self();
        }

        @Nonnull
        protected abstract B self();
    }
}
