package com.atlassian.bitbucket.event.commit;

import com.atlassian.bitbucket.commit.CommitDiscussion;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.event.api.AsynchronousPreferred;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.util.Set;

/**
 * An event raised when the participants in a commit discussion change. For example, when a user posts their first
 * comment on a commit, or replies to an existing comment, they are added as a participant and this event is raised.
 */
@AsynchronousPreferred
public class CommitDiscussionParticipantsUpdatedEvent extends CommitDiscussionEvent {

    private final Set<ApplicationUser> addedParticipants;
    private final Set<ApplicationUser> removedParticipants;

    public CommitDiscussionParticipantsUpdatedEvent(@Nonnull Object source, @Nonnull CommitDiscussion discussion,
                                                    @Nonnull Set<ApplicationUser> added, @Nonnull Set<ApplicationUser> removed) {
        super(source, discussion);

        this.addedParticipants = ImmutableSet.copyOf(added);
        this.removedParticipants = ImmutableSet.copyOf(removed);
    }

    /**
     * @return the set of users who have participated in the pull request for the first time
     */
    @Nonnull
    public Set<ApplicationUser> getAddedParticipants() {
        return addedParticipants;
    }

    /**
     * @return the set of users who are no longer participating in the pull request
     */
    @Nonnull
    public Set<ApplicationUser> getRemovedParticipants() {
        return removedParticipants;
    }
}
