package com.atlassian.bitbucket.event.cluster;

import com.atlassian.bitbucket.cluster.ClusterNode;
import com.atlassian.event.api.AsynchronousPreferred;

import javax.annotation.Nonnull;
import java.util.Date;
import java.util.Set;

/**
 * Specialization of {@link ClusterNodeAddedEvent} that is raised when a node reconnects to the current node. This
 * happens when a network partition is resolved, for instance due to intermittent networking issues.
 * <p>
 * Note: when a network partition in a larger cluster occurs, the node may receive multiple
 * {@code ClusterNodeRejoinedEvent}s when the partitions merge - one for each node-to-node connection that is
 * reestablished.
 */
@AsynchronousPreferred
public class ClusterNodeRejoinedEvent extends ClusterNodeAddedEvent {

    private final long timestampReconnected;
    private final long timestampDisconnected;

    public ClusterNodeRejoinedEvent(@Nonnull Object source, @Nonnull ClusterNode node,
                                    @Nonnull Set<ClusterNode> currentNodes,
                                    long timestampDisconnected, long timestampReconnected) {
        super(source, node, currentNodes);

        this.timestampReconnected = timestampReconnected;
        this.timestampDisconnected = timestampDisconnected;
    }

    /**
     * @return the date the node was disconnected from the current node
     */
    @Nonnull
    public Date getDateDisconnected() {
        return new Date(timestampDisconnected);
    }

    /**
     * @return the date the node reconnected to the current node
     */
    @Nonnull
    public Date getDateReconnected() {
        return new Date(timestampReconnected);
    }

    @Override
    public boolean isMaybeNetworkPartitionResolved() {
        return true;
    }
}
