package com.atlassian.bitbucket.event.cluster;

import com.atlassian.bitbucket.cluster.ClusterNode;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.util.Set;

import static java.util.Objects.requireNonNull;

/**
 * A base class for events related to cluster membership, raised when nodes {@link ClusterNodeAddedEvent join} and
 * {@link ClusterNodeRemovedEvent leave} the cluster.
 * <p>
 * This class exists primarily to simplify writing event types. Plugins generally should not listen for this base
 * class; they should listen for specific subclasses.
 */
public abstract class ClusterMembershipEvent extends ClusterNodeEvent {

    private final Set<ClusterNode> currentNodes;

    ClusterMembershipEvent(@Nonnull Object source, @Nonnull ClusterNode node, @Nonnull Set<ClusterNode> currentNodes) {
        super(source, node);

        this.currentNodes = ImmutableSet.copyOf(requireNonNull(currentNodes, "currentNodes"));
    }

    /**
     * Retrieves the <i>current</i> set of {@link ClusterNode cluster nodes} at the time the event was raised. If
     * a new node has joined, it will be in the returned set. If an existing node has departed, it will not be.
     *
     * @return the <i>current</i> cluster nodes
     */
    @Nonnull
    public Set<ClusterNode> getCurrentNodes() {
        return currentNodes;
    }
}
