package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.repository.Repository;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Instant;

/**
 * Defines a request to retrieve commits which are ancestors of a given {@link #getCommitId() starting commit}.
 */
public class CommitsRequest extends AbstractCommitsRequest {

    private final String commitId;
    private final Instant since;

    private CommitsRequest(Builder builder) {
        super(builder);

        commitId = builder.commitId;
        since = builder.since;
    }

    /**
     * @return the ID of the commit to load, which may be a branch or tag name or a {@link Commit#getId() commit ID}
     */
    @Nonnull
    public String getCommitId() {
        return commitId;
    }

    /**
     * @return the date since when commits are requested from or {@code null} if none is set
     * @since 7.12
     */
    @Nullable
    public Instant getSince() {
        return since;
    }

    public static class Builder extends AbstractCommitsRequestBuilder<Builder> {

        private final String commitId;
        private Instant since;

        public Builder(@Nonnull Repository repository, @Nonnull String commitId) {
            super(repository);

            this.commitId = requireNonBlank(commitId, "commitId");
        }

        /**
         * @param value the date since when commits are requested from
         * @return {@code this}
         * @since 7.12
         */
        @Nonnull
        public Builder since(@Nullable Instant value) {
            since = value;

            return this;
        }

        @Nonnull
        public CommitsRequest build() {
            return new CommitsRequest(this);
        }

        @Override
        protected Builder self() {
            return this;
        }
    }
}
