package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableList;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * Defines a request to retrieve changesets for a given set of commits. If any of the specified commits have multiple
 * {@link Commit#getParents() parents}, the changeset retrieved will be relative to the <i>first parent</i>.
 */
public class ChangesetsRequest {

    private static final int UNLIMITED_MESSAGE_LENGTH = -1;

    private final List<String> commitIds;
    private final boolean ignoreMissing;
    private final int maxChangesPerCommit;
    private final int maxMessageLength;
    private final Repository repository;

    private ChangesetsRequest(Builder builder) {
        commitIds = builder.commitIds.build();
        ignoreMissing = builder.ignoreMissing;
        maxChangesPerCommit = builder.maxChangesPerCommit;
        repository = builder.repository;
        maxMessageLength = builder.maxMessageLength;

        if (commitIds.isEmpty()) {
            throw new IllegalStateException(
                    "At least one commit ID must be provided to retrieve changesets");
        }
        if (maxChangesPerCommit < 1) {
            throw new IllegalStateException(
                    "The maximum number of changes to include per changeset must be greater than 0");
        }
    }

    /**
     * @return a collection containing one or more commit IDs to build changesets for
     */
    @Nonnull
    public Iterable<String> getCommitIds() {
        return commitIds;
    }

    /**
     * @return the maximum size of a commit message will be read, anything larger should be truncated
     */
    public int getMaxMessageLength() {
        return maxMessageLength;
    }

    /**
     * @return the maximum number of changes to include when building changesets for found commits
     */
    public int getMaxChangesPerCommit() {
        return maxChangesPerCommit;
    }

    /**
     * @return the repository to find the commits in
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    /**
     * @return whether this request should ignore missing commits, defaulting to {@code false}
     */
    public boolean isIgnoreMissing() {
        return ignoreMissing;
    }

    public static class Builder extends BuilderSupport {

        private final ImmutableList.Builder<String> commitIds;
        private final Repository repository;

        private boolean ignoreMissing;
        private int maxChangesPerCommit;
        private int maxMessageLength;

        public Builder(@Nonnull Repository repository) {
            this.repository = requireNonNull(repository, "repository");

            commitIds = ImmutableList.builder();
            maxMessageLength = UNLIMITED_MESSAGE_LENGTH;
            maxChangesPerCommit = 5;
        }

        @Nonnull
        public ChangesetsRequest build() {
            return new ChangesetsRequest(this);
        }

        @Nonnull
        public Builder commitId(@Nullable String value) {
            addIf(NOT_BLANK, commitIds, value);

            return this;
        }

        @Nonnull
        public Builder commitIds(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, commitIds, value, values);

            return this;
        }

        @Nonnull
        public Builder commitIds(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, commitIds, values);

            return this;
        }

        @Nonnull
        public Builder ignoreMissing(boolean value) {
            ignoreMissing = value;

            return this;
        }

        @Nonnull
        public Builder maxChangesPerCommit(int value) {
            checkArgument(value > 0, "At least one change must be included for each changeset");
            maxChangesPerCommit = value;

            return this;
        }

        @Nonnull
        public Builder messageLength(int maxMessageLength) {
            this.maxMessageLength = maxMessageLength;

            return this;
        }
    }
}
