package com.atlassian.bitbucket.comment;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

import javax.annotation.Nonnull;
import java.util.Deque;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import static java.util.Objects.requireNonNull;

/**
 * Performs a depth-first iteration given a starting set of {@link Comment comments}. For example, consider the
 * following comments:
 * <pre>
 * C1
 *   C1-R1
 *   C1-R2
 * C2
 *   C2-R1
 *     C2-R1-R1
 *   C2-R2
 *     C2-R2-R1
 *       C2-R2-R1-R1
 *   C2-R3
 * C3
 * </pre>
 * {@code new CommentChain(ImmutableList.of(c1, c2, c3))} would traverse the comments in the same order as reading
 * that tree from the top to the bottom:
 * <code>
 * C1 -&gt; C1-R1 -&gt; C1-R2 -&gt; C2 -&gt; C2-R1 -&gt; C2-R1-R1 -&gt; C2-R2 -&gt; C2-R2-R1 -&gt; C2-R2-R1-R1 -&gt; C2-R3 -&gt; C3
 * </code>
 * Notice that C2-R1-R1, the first reply to the first reply of the second comment, is traversed before C2-R2, the
 * second reply to the second comment.
 * <p>
 * Note: This class supports any subtype {@code C} of {@link Comment}, under the assumption that all
 * {@link Comment#getComments() replies} are consistently instances of {@code C}. Violating that assumption will
 * result in {@code ClassCastException}s when iterating through the chain.
 */
public class CommentChain<C extends Comment> implements Iterable<C> {

    private final List<C> rootComments;

    /**
     * @param rootComments list of comment thread roots to expand
     */
    private CommentChain(@Nonnull Iterable<C> rootComments) {
        this.rootComments = ImmutableList.copyOf(requireNonNull(rootComments, "rootComments"));
    }

    /**
     * A convenience method for creating a new {@link CommentChain chain} from the provided comments.
     *
     * @param rootComments the comments to use when creating the chain
     * @param <C> an implementation of {@link Comment}
     * @return a new chain
     * @since 5.0
     */
    public static <C extends Comment> CommentChain<C> of(@Nonnull Iterable<C> rootComments) {
        return new CommentChain<>(rootComments);
    }

    /**
     * @return stream of the comment chain
     * @since 4.1
     */
    @Nonnull
    public Stream<C> stream() {
        return StreamSupport.stream(spliterator(), false);
    }

    @Override
    @Nonnull
    public Iterator<C> iterator() {
        return new DepthFirstIterator();
    }

    private class DepthFirstIterator implements Iterator<C> {

        private final Deque<C> commentStack = new LinkedList<>();

        DepthFirstIterator() {
            pushAll(rootComments);
        }

        @Override
        public boolean hasNext() {
            return !commentStack.isEmpty();
        }

        @Override
        @SuppressWarnings("unchecked")
        public C next() {
            C currentComment = commentStack.pop();
            // As described in the Javadoc, this will _only_ work if the replies are instances of C too, otherwise
            // pushAll will throw a ClassCastException
            pushAll((List) currentComment.getComments());

            return currentComment;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("Element removal is not supported by " + getClass().getName());
        }

        private void pushAll(List<C> comments) {
            for (C comment : Lists.reverse(comments)) {
                commentStack.push(comment);
            }
        }
    }
}
