package com.atlassian.bitbucket.comment;

import com.atlassian.bitbucket.content.DiffFileType;
import com.atlassian.bitbucket.content.DiffSegmentType;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Objects.requireNonNull;

/**
 * Request to create or update a {@link CommentThreadDiffAnchor#isLineAnchor() line comment}.
 *
 * @since 5.0
 */
public class AddLineCommentRequest extends AddFileCommentRequest {

    private final DiffFileType fileType;
    private final int line;
    private final DiffSegmentType lineType;

    private AddLineCommentRequest(Builder builder) {
        super(builder);

        checkArgument(builder.line != CommentThreadDiffAnchor.NO_LINE, "line");

        this.fileType = requireNonNull(builder.fileType, "fileType");
        this.line = builder.line;
        this.lineType = requireNonNull(builder.lineType, "lineType");
    }

    /**
     * @return the type of the file the comments is attached to
     */
    @Nonnull
    public DiffFileType getFileType() {
        return fileType;
    }

    /**
     * @return the line number in the diff
     */
    public int getLine() {
        return line;
    }

    /**
     * @return the type of the diff segment the commented line represents
     */
    @Nonnull
    public DiffSegmentType getLineType() {
        return lineType;
    }

    public static class Builder extends AbstractFileBuilder<Builder, AddLineCommentRequest> {

        private int line = CommentThreadDiffAnchor.NO_LINE;
        private DiffSegmentType lineType;
        private DiffFileType fileType;

        public Builder(@Nonnull Commentable commentable, @Nonnull String text,
                       @Nonnull CommentThreadDiffAnchorType diffType, @Nonnull String path) {
            super(commentable, text, diffType, path);
        }

        @Nonnull
        @Override
        public AddLineCommentRequest build() {
            return new AddLineCommentRequest(this);
        }

        @Nonnull
        public Builder fileType(@Nonnull DiffFileType value) {
            fileType = requireNonNull(value, "fileType");

            return this;
        }

        @Nonnull
        public Builder line(int value) {
            checkArgument(value > CommentThreadDiffAnchor.NO_LINE, "Line must be > %s (Rejected: %s)",
                    CommentThreadDiffAnchor.NO_LINE, value);
            line = value;

            return this;
        }

        @Nonnull
        public Builder lineType(@Nonnull DiffSegmentType value) {
            lineType = requireNonNull(value, "lineType");

            return this;
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
