package com.atlassian.beehive.simple;

import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

import javax.annotation.Nonnull;

import com.atlassian.beehive.core.ClusterLockStatus;
import com.atlassian.beehive.core.ManagedClusterLock;
import com.atlassian.beehive.core.stats.StatisticsKey;

import com.google.common.collect.ImmutableMap;

import static com.atlassian.beehive.core.stats.StatisticsKey.WAIT_QUEUE_LENGTH;
import static java.util.Objects.requireNonNull;

/**
 * An implementation of ClusterLockStatus that is only suitable for single-node applications.
 */
@SuppressWarnings("LockAcquiredButNotSafelyReleased")
class SingleNodeLock implements ManagedClusterLock
{
    private final String name;
    private final ReentrantLock lock = new ReentrantLock();
    private static final String NODE_NAME = "single node";

    SingleNodeLock(final String name)
    {
        this.name = requireNonNull(name, "name");
    }

    @Nonnull
    @Override
    public String getName()
    {
        return name;
    }

    @Override
    public boolean isLocked() {
        return lock.isLocked();
    }

    @Override
    public void lockInterruptibly() throws InterruptedException
    {
        lock.lockInterruptibly();
    }

    @Override
    public boolean tryLock()
    {
        return lock.tryLock();
    }

    @Override
    public boolean tryLock(long timeout, @Nonnull TimeUnit unit) throws InterruptedException
    {
        return lock.tryLock(timeout, unit);
    }

    @Override
    public void lock()
    {
        lock.lock();
    }

    @Override
    public void unlock()
    {
        lock.unlock();
    }

    @Override
    public boolean isHeldByCurrentThread()
    {
        return lock.isHeldByCurrentThread();
    }

    @Nonnull
    @Override
    public Condition newCondition()
    {
        // Because the database impl can't
        throw new UnsupportedOperationException("newCondition() not supported in ClusterLock");
    }


    /**
     * {@inheritDoc}
     * <p>
     * The single node lock only reports the lock's {@link StatisticsKey#WAIT_QUEUE_LENGTH}.
     * No other statistics are supported at this time.
     * </p>
     */
    @Nonnull
    @Override
    public Map<StatisticsKey,Long> getStatistics()
    {
        final int queueLen = lock.getQueueLength();
        if (queueLen > 0)
        {
            return ImmutableMap.of(WAIT_QUEUE_LENGTH, (long)queueLen);
        }
        return ImmutableMap.of();
    }

    @Nonnull
    @Override
    public ClusterLockStatus getClusterLockStatus() {
        return new ClusterLockStatus(this.getName(), NODE_NAME, 0);
    }
}
