package com.atlassian.bamboo.plugins.docker.export;

import com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator;
import com.atlassian.bamboo.specs.api.builders.task.Task;
import com.atlassian.bamboo.specs.api.model.task.TaskProperties;
import com.atlassian.bamboo.specs.api.validators.common.ValidationProblem;
import com.atlassian.bamboo.specs.builders.task.AbstractDockerRegistryTask;
import com.atlassian.bamboo.specs.builders.task.DockerPullImageTask;
import com.atlassian.bamboo.specs.builders.task.DockerPushImageTask;
import com.atlassian.bamboo.specs.model.task.docker.AbstractDockerTaskProperties;
import com.atlassian.bamboo.specs.model.task.docker.DockerRegistryTaskProperties;
import com.atlassian.bamboo.task.TaskContainer;
import com.atlassian.bamboo.task.TaskDefinition;
import com.atlassian.bamboo.task.export.TaskDefinitionExporter;
import com.atlassian.bamboo.task.export.TaskValidationContext;
import com.atlassian.bamboo.util.Narrow;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.DOCKER_COMMAND_OPTION;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.DOCKER_COMMAND_OPTION_PULL;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.DOCKER_COMMAND_OPTION_PUSH;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.EMAIL;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.PASSWORD;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.PULL_EMAIL;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.PULL_PASSWORD;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.PULL_REPOSITORY;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.PULL_USERNAME;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.PUSH_REPOSITORY;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.REGISTRY_OPTION;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.REGISTRY_OPTION_CUSTOM;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.REGISTRY_OPTION_HUB;
import static com.atlassian.bamboo.plugins.docker.tasks.cli.DockerCliTaskConfigurator.USERNAME;
import static com.atlassian.bamboo.task.TaskConfigConstants.CFG_ENVIRONMENT_VARIABLES;
import static com.atlassian.bamboo.task.TaskConfigConstants.CFG_WORKING_SUBDIRECTORY;

@Component
public class RegistryTaskExporter implements TaskDefinitionExporter {

    private static final Set<String> ACCEPTABLE_COMMAND_OPTIONS = ImmutableSet.of(DOCKER_COMMAND_OPTION_PULL, DOCKER_COMMAND_OPTION_PUSH);

    @NotNull
    @Override
    public Map<String, String> toTaskConfiguration(@NotNull TaskContainer taskContainer, @NotNull TaskProperties taskProperties) {
        DockerRegistryTaskProperties properties = Narrow.downTo(taskProperties, DockerRegistryTaskProperties.class);
        Preconditions.checkArgument(properties != null,
                "Don't now how to import " + taskProperties.getClass().getCanonicalName());

        ImmutableMap.Builder<String, String> config = ImmutableMap.builder();

        switch (properties.getRegistryType()) {
            case DOCKER_HUB:
                config.put(REGISTRY_OPTION, REGISTRY_OPTION_HUB);
                break;
            case CUSTOM:
                config.put(REGISTRY_OPTION, REGISTRY_OPTION_CUSTOM);
                break;
            default:
                throw new IllegalStateException("Couldn't import: " + taskProperties);
        }

        switch (properties.getOperationType()) {
            case PUSH:
                config.put(DOCKER_COMMAND_OPTION, DOCKER_COMMAND_OPTION_PUSH);
                config.put(DockerCliTaskConfigurator.EMAIL, StringUtils.defaultString(properties.getEmail()));
                config.put(DockerCliTaskConfigurator.USERNAME, StringUtils.defaultString(properties.getUsername()));
                config.put(DockerCliTaskConfigurator.PASSWORD, StringUtils.defaultString(properties.getPassword()));
                config.put(DockerCliTaskConfigurator.PUSH_REPOSITORY, StringUtils.defaultString(properties.getImage()));
                break;
            case PULL:
                config.put(DOCKER_COMMAND_OPTION, DOCKER_COMMAND_OPTION_PULL);
                config.put(DockerCliTaskConfigurator.PULL_EMAIL, StringUtils.defaultString(properties.getEmail()));
                config.put(DockerCliTaskConfigurator.PULL_USERNAME, StringUtils.defaultString(properties.getUsername()));
                config.put(DockerCliTaskConfigurator.PULL_PASSWORD, StringUtils.defaultString(properties.getPassword()));
                config.put(DockerCliTaskConfigurator.PULL_REPOSITORY, StringUtils.defaultString(properties.getImage()));
                break;
            default:
                throw new IllegalStateException("Couldn't handle operation type: " + properties.getOperationType());
        }

        config.put(CFG_ENVIRONMENT_VARIABLES, StringUtils.defaultString(properties.getEnvironmentVariables()));
        config.put(CFG_WORKING_SUBDIRECTORY, StringUtils.defaultString(properties.getWorkingSubdirectory()));

        return config.build();
    }

    @NotNull
    @Override
    public Task toSpecsEntity(@NotNull TaskDefinition taskDefinition) {
        if (isDockerRegistryTask(taskDefinition)) {
            Map<String, String> config = taskDefinition.getConfiguration();
            switch (config.getOrDefault(DOCKER_COMMAND_OPTION, "")) {

                case DOCKER_COMMAND_OPTION_PUSH:
                    DockerPushImageTask pushTask = new DockerPushImageTask()
                            .authentication(config.getOrDefault(USERNAME, ""),
                                    config.getOrDefault(PASSWORD, ""),
                                    config.getOrDefault(EMAIL, ""))
                            .environmentVariables(config.getOrDefault(CFG_ENVIRONMENT_VARIABLES, ""))
                            .workingSubdirectory(config.getOrDefault(CFG_WORKING_SUBDIRECTORY, ""));

                    return setDockerImage(pushTask, config.getOrDefault(REGISTRY_OPTION, ""), config.getOrDefault(PUSH_REPOSITORY, ""));

                case DOCKER_COMMAND_OPTION_PULL:
                    DockerPullImageTask pullTask = new DockerPullImageTask()
                            .dockerHubImage(config.getOrDefault(PULL_REPOSITORY, ""))
                            .authentication(config.getOrDefault(PULL_USERNAME, ""),
                                    config.getOrDefault(PULL_PASSWORD, ""),
                                    config.getOrDefault(PULL_EMAIL, ""))
                            .environmentVariables(config.getOrDefault(CFG_ENVIRONMENT_VARIABLES, ""))
                            .workingSubdirectory(config.getOrDefault(CFG_WORKING_SUBDIRECTORY, ""));

                    return setDockerImage(pullTask, config.getOrDefault(REGISTRY_OPTION, ""), config.getOrDefault(PULL_REPOSITORY, ""));

                default:
                    throw new IllegalStateException("Something went badly. Couldn't export.");
            }
        }

        throw new IllegalArgumentException(String.format("Couldn't export task by id: %s and plugin key: %s", taskDefinition.getId(), taskDefinition.getPluginKey()));
    }

    @NotNull
    private Task setDockerImage(AbstractDockerRegistryTask task, String registryOption, String registry) {
        switch (registryOption) {
            case REGISTRY_OPTION_HUB:
                return task.dockerHubImage(registry);
            case REGISTRY_OPTION_CUSTOM:
                return task.customRegistryImage(registry);
            default:
                throw new IllegalStateException("Something went badly. Couldn't export.");
        }
    }

    private boolean isDockerRegistryTask(@NotNull TaskDefinition taskDefinition) {
        return taskDefinition.getPluginKey().equals(AbstractDockerTaskProperties.MODULE_KEY.getCompleteModuleKey())
                && ACCEPTABLE_COMMAND_OPTIONS.contains(taskDefinition.getConfiguration().get(DOCKER_COMMAND_OPTION));
    }

    @Override
    public List<ValidationProblem> validate(@NotNull TaskValidationContext taskValidationContext, @NotNull TaskProperties taskProperties) {
        return Collections.emptyList();
    }
}
