define('codedeploy/aws-autocomplete', [
    'jquery',
    'underscore',
    'util/ajax',
    'widget/autocomplete',
    'credentials/dialog-form-add-credentials'
], function(
    $,
    _,
    ajax,
    Autocomplete,
    DialogFormAddCredentials
) {

    'use strict';

    var AWSAutocomplete = Backbone.View.extend({

        defaultSettings: {
            urlBuckets: _.template(AJS.contextPath() + '/rest/aws/latest/credentials/<%=credentialId%>/s3/buckets'),
            urlApplications: _.template(AJS.contextPath() + '/rest/aws/latest/credentials/<%=credentialId%>/regions/<%=region%>/codedeploy/applications'),
            urlDeploymentGroups: _.template(AJS.contextPath() + '/rest/aws/latest/credentials/<%=credentialId%>/regions/<%=region%>/codedeploy/applications/<%=application%>/deploymentGroups'),
            $credentialsEl: undefined,
            $regionEl: undefined,
            $bucketsEl: undefined,
            $applicationsEl: undefined,
            $deploymentGroupsEl: undefined
        },

        initialize: function(settings) {
            _.extend(this, this.defaultSettings, settings);

            this.bucketsAutocomplete = this.createAutocomplete(this.$bucketsEl, 'bucketsAutocomplete');
            this.applicationsAutocomplete = this.createAutocomplete(this.$applicationsEl, 'applicationsAutocomplete');
            this.deploymentGroupsAutocomplete = this.createAutocomplete(this.$deploymentGroupsEl, 'deploymentGroupsAutocomplete');

            this.attachEvents();
            this.populateAutocomplete();
        },

        createAutocomplete: function(el, namespace) {
            return new Autocomplete({
                el: el,
                formatResult: this.onAutocompleteFormat,
                formatSelection: this.onAutocompleteFormat,
                query: _.bind(this.onQuery, this, namespace)
            })
        },

        attachEvents: function(){
            $(this.$credentialsEl).add(this.$regionEl).change(_.bind(function(){
                [this.bucketsAutocomplete, this.applicationsAutocomplete, this.deploymentGroupsAutocomplete].forEach(function(autocomplete){
                    autocomplete.clearValue();
                    autocomplete.processData([]);
                }, this);
                this.populateAutocomplete();
            }, this));

            this.$applicationsEl.change(_.bind(this.populateDeploymentGroups, this));

            this.$addCredentialsEl.click(_.bind(function(evt){
                evt.preventDefault();
                new DialogFormAddCredentials({
                    $credentialsEl: this.$credentialsEl
                });
            }, this));
        },

        onQuery: function(autocomplete, query){
            query.callback({
                results: _.filter(this[autocomplete].data, function(e) {
                    return query.text === '' || e.text.toUpperCase().indexOf(query.term.toUpperCase()) >= 0;
                })
            });
        },

        onAutocompleteFormat: function(item) {
            return item.text;
        },

        populateAutocomplete: function(){
            this.populateBuckets();
            this.populateApplications();
        },

        populateBuckets: function(){
            var credentialId = this.$credentialsEl.val();
            if (!credentialId){
                return;
            }

            return this.fetchResults(this.bucketsAutocomplete, this.urlBuckets({
                credentialId: credentialId
            }));
        },

        populateApplications: function(){
            var credentialId = this.$credentialsEl.val();
            if (!credentialId){
                return;
            }

            return this.fetchResults(this.applicationsAutocomplete, this.urlApplications({
                credentialId: credentialId,
                region: this.$regionEl.val()
            }));
        },

        populateDeploymentGroups: function(){
            var applicationName = this.$applicationsEl.val();
            if (!applicationName){
                return;
            }

            return this.fetchResults(this.deploymentGroupsAutocomplete, this.urlDeploymentGroups({
                credentialId: this.$credentialsEl.val(),
                region: this.$regionEl.val(),
                application: applicationName
            }));
        },

        fetchResults: function(autocomplete, url){
            autocomplete.$el.parent().find('.ajax-error, .aui-icon-wait').remove();
            autocomplete.addLoadingIcon();
            autocomplete.disable();
            return ajax({ url: url, error: {}})
                .then(_.bind(this.onRequestSuccess, autocomplete))
                .fail(_.bind(this.onRequestError, autocomplete))
                .always(_.bind(this.onRequestEnd, autocomplete));
        },

        onRequestSuccess: function(data) {
            var results = data.results.map(function(result){
                return {
                    text: result.name,
                    id: result.name
                };
            });

            var selectValue = results[0];
            var oldValue = this.$el.val();
            if (oldValue) {
                selectValue = results.filter(function (result) {
                    return result.text === oldValue;
                }, this)[0] || selectValue;
            }

            this.$el.auiSelect2('data', selectValue);
            this.$el.trigger('change');
            this.processData(results);
        },

        onRequestError: function(data) {
            if (data.responseJSON && data.responseJSON.message) {
                this.$el.after($(bamboo.plugins.aws.autocomplete.error({
                    message: data.responseJSON.message
                })));
            }
            this.clearValue();
            this.processData([]);
        },

        onRequestEnd: function() {
            this.removeLoadingIcon();
            this.enable();
        }

    });

    return AWSAutocomplete;
});
