package com.atlassian.application.api;

import com.atlassian.fugue.Option;
import com.atlassian.sal.api.user.UserKey;

import java.net.URI;
import javax.annotation.Nullable;

/**
 * Represents the access control to a {@link Application} (e.g. Service Desk) installed on top of a platform
 * (e.g. JIRA). Note, this is not a specific licensing check, as long as there is a license that can grant access
 * present then there will be no access errors. That is, the access may not come from the license of the specific
 * application being checked.
 *
 * Unless otherwise noted none of the parameters or return values can be {@code null}.
 *
 * @since v1.0
 */
public interface ApplicationAccess
{
    /**
     * Represents the different types of access error.
     */
    enum AccessError
    {
        /**
         * The {@link Application} is not licensed.
         */
        UNLICENSED,

        /**
         * The {@link Application} is licensed but that license is now out of date. Some licenses do not have an expiry.
         */
        EXPIRED,

        /**
         * The {@link Application} is too new for the license.
         */
        VERSION_MISMATCH,

        /**
         * The {@link Application} is licensed but too many users are allowed access.
         */
        USERS_EXCEEDED,

        /**
         * The {@link Application} is licensed but the user does not have access.
         */
        NO_ACCESS
    }

    /**
     * Get the {@link com.atlassian.application.api.ApplicationKey} of the {@link Application} whose access control
     * is managed by this instance.
     *
     * @return the {@code ApplicationKey} of the {@code Application} whose access control is managed by this instance.
     */
    ApplicationKey getApplicationKey();

    /**
     * Return the maximum number of users that can be given access to the {@link Application}.
     * {@link com.atlassian.fugue.Option#none()} is returned to indicate there is no maximum.
     *
     * @return the maximum number of users that can be given access to the application.
     * {@link com.atlassian.fugue.Option#none()} is returned to indicate there is no maximum.
     */
    Option<Integer> getMaximumUserCount();

    /**
     * Returns the number of users who are currently able to access the associated {@link Application}.
     * <p>
     *     This function may be called frequently; it is strongly recommended that implementations use a caching
     *     strategy to reduce the cost of repeated calls.
     *
     * @return the number of users configured for access to the associated {@link Application}.
     */
    int getActiveUserCount();

    /**
     * Returns {@code true} if the user actually able to use the associated {@link Application}. This method returns
     * {@code true} when {@code getAccessError().isEmpty()} is {@code true}.
     *
     * <p>
     *     This function may be called frequently; it is strongly recommended that implementations use a caching
     *     strategy to reduce the cost of repeated calls.
     *
     * @param userKey the user key. {@code null} represents the anonymous user.
     * @return true if the given user can access the associated {@link Application}.
     */
    boolean canUserAccessApplication(@Nullable UserKey userKey);

    /**
     * Returns the most important reason why the passed user cannot access the
     * {@link Application}. {@link com.atlassian.fugue.Option#none()} will be returned
     * if the user can access the {@code Application}.
     *
     * @param userKey the user key. {@code null} represents the anonymous user.
     * @return The reason why the user cannot access the {@code Application} or
     * {@link com.atlassian.fugue.Option#none()} if the user is able to access the {@code Product}.
     */
    Option<AccessError> getAccessError(@Nullable UserKey userKey);

    /**
     * Returns a relative URI to a screen where administrators can manage application access.
     * <p>
     *     The URI must reference a page on the current application instance.
     * <p>
     *     This URI should not include the application's context path if one exists.
     *
     * @return a relative URI to a screen where administrators can manage users in this application.
     */
    URI getManagementPage();
}
