(function ($) {
    // Make sure we are getting the jQuery.ajax method, even if it has been overridden elsewhere.
    // This will only work if this code is executed before any code which overrides the ajax method.
    var $ajax = AJS.$.ajax;
    var baseStorageKey = 'atlassian-analytics';
    var contextPath =
        typeof AJS.contextPath === "function" ? AJS.contextPath() : "";

    var publish = null;
    var storageKey = null;
    var lockKey = null;

    // A unique identifier for this browser tab
    // Source: http://stackoverflow.com/a/2117523
    var uuid = 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
        var r = Math.random()*16|0, v = c == 'x' ? r : (r&0x3|0x8);
        return v.toString(16);
    });

    var determineStorageKey = function(){
        // We need to give each product its own key for events because it's possible to host multiple products
        // at the same url.
        var product = 'unknown';
        if (document.body.id == 'jira'){
            product = 'jira';
        }
        else if (document.body.id == 'com-atlassian-confluence'){
            product = 'confluence';
        }
        storageKey = baseStorageKey + '.' + product;
        lockKey = storageKey + '.lock';
    };

    var getLock = function() {
        if (store.get(lockKey)) {
            return false;
        }

        store.set(lockKey, uuid);

        // Reduce chance of race condition - read back lock to make sure we still have it
        return (store.get(lockKey) === uuid);
    };

    var releaseLock = function() {
        store.set(lockKey, null);
    };

    /**
     * Persists the events that have been generated until such time that they can be sent.
     */
    var saveForLater = function() {
        var events = [],
            event,
            e, i, ii;
        if (AJS.EventQueue.length == 0)
            return;
        // Prime our events array with anything that's already saved.
        events = store.get(storageKey) || events;
        // Suck the events out of the event queue and in to our events array.
        for (i = 0, ii = AJS.EventQueue.length; i < ii; ++i) {
            e = AJS.EventQueue[i];
            if (e.name) {
                // the queue could contain anything - shear unusable properties
                event = { name: e.name, properties: e.properties, time: e.time || 0};
                events.push(event);
            }
        }
        // Empty the event queue
        AJS.EventQueue.length = 0;
        // Save our events for later
        store.set(storageKey, events);
    };

    // Variable to track the number of retries to publish
    var bulkPublishRetryCount = 0;

    /**
     * Gets the amount of time that should be waited until the next publish attempt.
     * @param retryCount How many requests failed since the last successful publish.
     * @returns {number} how many ms that should be waited.
     */
    var getBulkPublishBackoff = function (retryCount) {
        return Math.min(5000 * Math.pow(2, retryCount), 5*60*1000);
    };

    var lockFailures = 0;

    /**
     * Publishes every event that's ready for publishing.
     */
    var bulkPublish = function() {
        var events;

        function reschedule() {
            setTimeout(bulkPublish, getBulkPublishBackoff(bulkPublishRetryCount = 0));
        }

        function rescheduleFailed() {
            setTimeout(bulkPublish, getBulkPublishBackoff(++bulkPublishRetryCount));
        }

        // Avoid multiple browser tabs hitting this all at once
        if (!getLock()) {
            ++lockFailures;
            // if we have been failing to get the lock for a while, just grab it
            if (lockFailures < 20) {
                return reschedule();
            }
        } else {
            lockFailures = 0;
        }
        try {
            // Make sure every event we might have is stored.
            saveForLater();
            // Pull the stored events out and get 'em ready for transmission.
            events = store.get(storageKey);

            if (!events || !events.length) {
                return reschedule();
            }

            // Wipe the stored events.
            store.remove(storageKey);

            // Validate events and remove any dodgy ones
            if (!validateEvents(events)) {
                return reschedule();
            }

            // try to present a rough timing of events that the server can interpret relative to it's own time.
            var currentTime = new Date().getTime();
            for (var i = 0; i < events.length; i++) {
                if (events[i].time > 0) {
                    events[i].timeDelta = events[i].time - currentTime;
                }
                else {
                    // just fake it. This corresponds to a millisecond for each place behind last in the array.
                    // This should be rare. Basically, events added to EventQueue before this script was loaded.
                    events[i].timeDelta = i - events.length;
                }
                delete events[i].time;
            }

            // AJS.safe.post appears to corrupt a JSON data object, so we send it as a context param instead.
            // Failing to JSON encode the data results in jQuery not attempting to send, and silently swallowing our attempt
            publish = $ajax({
                type: "POST",
                url: contextPath + "/rest/analytics/1.0/publish/bulk",
                data: JSON.stringify(events),
                contentType: "application/json",
                dataType: "json"
            });
            // In case the transmission fails, let's keep the events we just attempted to send.
            publish.fail(function () {
                // This actually drops events, but the alternative is to use something like:
                //   $.merge(AJS.EventQueue, events);
                // Unfortunately using that will cause some fairly nasty issues where duplicate events continually
                // get sent - see https://jira.atlassian.com/browse/AA-179 for more details.
                // TODO: investigate why the above happens and fix this functionality for good
                AJS.EventQueue.concat(events);

                saveForLater();
                rescheduleFailed();
            });
            publish.done(function () {
                reschedule();
            });
        } finally {
            releaseLock();
        }
    };

    /**
     * Check for any invalid events and remove/sanitise them.
     * @param events - the list of events to be published
     * @returns the number of valid events remaining
     */
    var validateEvents = function(events) {
        for (var i = events.length - 1; i >= 0; i--) {
            var validMsg = "";
            var event = events[i];
            var properties = event.properties;
            if (typeof event.name === "undefined") {
                validMsg = "you must provide a name for the event.";
            } else if (typeof properties !== "undefined" && properties !== null) {
                if (properties.constructor !== Object) {
                    validMsg = "properties must be an object with key value pairs.";
                } else {
                    sanitiseProperties(properties);
                }
            }
            if (validMsg !== "") {
                AJS.log("WARN: Invalid analytics event detected and ignored, " + validMsg + "\nEvent: "+JSON.stringify(event));
                events.splice(i, 1);
            }
        }
        return events.length;
    };

    var sanitiseProperties = function(properties) {
        for (var propertyName in properties) {
            if (properties.hasOwnProperty(propertyName)) {
                var propertyValue = properties[propertyName];

                if (valueExists(propertyValue) && isAllowedType(propertyValue)) {
                    // Do nothing - the property value is safe & allowed already
                } else if (valueExists(propertyValue) && propertyValue.toString) {
                    // Sanitise the property value by invoking its "toString"
                    properties[propertyName] = propertyValue.toString();
                } else {
                    // If it's an undefined, null or invalid value - blank it out
                    properties[propertyName] = "";
                }
            }
        }
    };

    function valueExists(propertyValue) {
        return typeof propertyValue !== "undefined" && propertyValue !== null;
    }

    function isAllowedType(propertyValue) {
        return typeof propertyValue === "number" || typeof propertyValue === "string" || typeof propertyValue === "boolean";
    }

    var cancelPublish = function() {
        if (publish && !(publish.state() === "resolved" || publish.state() === "rejected")) {
            publish.abort(); // This will cancel the request to the server, and cause the events to be saved for later.
        }
    };

    /**
     * Provides a way to publish events asynchronously, without requiring AJS.Events to have loaded.
     * Users of this property must conditionally initialise it to an empty array. Objects pushed
     * must have a name property, and optionally a properties property of other data to send.
     * @example
     * AJS.EventQueue = AJS.EventQueue || [];
     * AJS.EventQueue.push({name: 'eventName', properties: {some: 'data', more: true, hits: 20}});
     */
    AJS.EventQueue = AJS.EventQueue || [];

    var arrayPush = Array.prototype.push;
    AJS.EventQueue.push = function(obj) {
    	obj.time = new Date().getTime();
    	arrayPush.call(AJS.EventQueue, obj);
    };

    AJS.toInit(function() {
    	determineStorageKey();
        setTimeout(bulkPublish, 500);
        removeOldAnalytics();
    });
    $(window).unload(function() {
        cancelPublish();
        saveForLater();
    });

    /**
     * @deprecated since v3.39, please trigger as normal and use whitelisting to denote privacy policy safe events
     */
    AJS.Analytics = {
        triggerPrivacyPolicySafeEvent: function(name, properties) {
            AJS.log("WARN: 'triggerPrivacyPolicySafeEvent' has been deprecated");
            AJS.EventQueue.push({name: name, properties: properties});
        }
    };

    /**
     * Binds to an event that developers can trigger without having to do any feature check.
     * If this code is available then the event will get published and if it's not the event
     * will go unnoticed.
     * @example
     * AJS.trigger('analytics', {name: 'pageSaved', data: {pageName: page.name, space: page.spaceKey}});
     */
    AJS.bind('analytics', function(event, data) {
    	AJS.EventQueue.push({name: data.name, properties: data.data});
    });

    // legacy binding until Confluence page layout JS is updated
    AJS.bind('analyticsEvent', function(event, data) {
    	AJS.EventQueue.push({name: data.name, properties: data.data});
    });

    /**
     * As part of bundling this plugin in BTF now, we need to remove the existing JIRA analytics setting if we see it.
     */
    var removeOldAnalytics = function () {
        if (window.location.pathname.indexOf("/secure/admin/ViewApplicationProperties") > -1) {
            AJS.$("[data-property-id='analytics-enabled']").remove();
        } else if (window.location.pathname.indexOf("/secure/admin/EditApplicationProperties") > -1) {
            var $analytics = AJS.$(":contains(Enable Atlassian analytics)");
            if ($analytics.size() > 0) {
                var parentElement = $analytics[$analytics.size() - 2];
                if (parentElement) {
                    parentElement.remove();
                }
            }
        }
    }

}(AJS.$));
