package com.atlassian.analytics.client.uuid;

import com.atlassian.analytics.client.ServerIdProvider;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;

import java.util.UUID;

public class ProductUUIDProvider {

    static final String ANALYTICS_UUID_KEY = "com.atlassian.analytics.client.configuration.uuid";
    static final String ANALYTICS_SERVERID_KEY = "com.atlassian.analytics.client.configuration.serverid";

    private final PluginSettingsFactory pluginSettingsFactory;
    private final ServerIdProvider serverIdProvider;

    public ProductUUIDProvider(PluginSettingsFactory pluginSettingsFactory, ServerIdProvider serverIdProvider) {
        this.pluginSettingsFactory = pluginSettingsFactory;
        this.serverIdProvider = serverIdProvider;
    }

    public String createUUID() {
        // ideally we'd put this through some kind of cluster-safe executor to ensure it only happens once per cluster.
        // This code needs to work for all products, so I can't make use of the BB ConcurrencyService.
        // If we wanted to make it cluster-safe, we'd have to use atlassian-beehive's ClusterLockService, which is
        // much lower level and I'd be afraid of causing deadlocks.
        // Currently there is an edge case where it could be called once in each node.
        // However, as long as we don't cache the value, it's eventually consistent. Given that we run through this
        // code on plugin start up, I'd expect the nodes to be in sync by the time we actually reading the value out and
        // applying them to events.

        String actualServerId = getActualServerId();

        if (!serverIdMatches(actualServerId)) {
            return generateUUIDForServerId(actualServerId);
        }

        String savedUUID = getSavedUUID();
        if (null != savedUUID) {
            return savedUUID;
        }

        return generateUUIDForServerId(actualServerId);
    }

    public String getUUID() {
        String uuid = getSavedUUID();
        if (null != uuid) {
            return uuid;
        }
        return createUUID();
    }

    private String generateUUIDForServerId(String actualServerId) {
        String newUuid = UUID.randomUUID().toString();
        pluginSettingsFactory.createGlobalSettings().put(ANALYTICS_UUID_KEY, newUuid);
        pluginSettingsFactory.createGlobalSettings().put(ANALYTICS_SERVERID_KEY, actualServerId);
        return newUuid;
    }

    private String getActualServerId() {
        return serverIdProvider.getServerId();
    }

    private String getSavedServerId() {
        return (String) pluginSettingsFactory.createGlobalSettings().get(ANALYTICS_SERVERID_KEY);
    }

    private String getSavedUUID() {
        return (String) pluginSettingsFactory.createGlobalSettings().get(ANALYTICS_UUID_KEY);
    }

    private boolean serverIdMatches(String actualServerId) {
        String lastServerId = getSavedServerId();
        return null == actualServerId ?
                null == lastServerId :
                actualServerId.equals(lastServerId);
    }
}