package com.atlassian.analytics.client.serialize;

import com.google.common.annotations.VisibleForTesting;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

/**
 * Additional information we want to grab from the request object
 * to add to serialized events.
 */
public class RequestInfo {
    @VisibleForTesting
    static final String X_FORWARDED_FOR_FIELD = "X-FORWARDED-FOR";
    @VisibleForTesting
    static final String ATL_PATH_COOKIE_NAME = "__atl_path";

    private String sourceIp;
    private String atlPath;
    private String B3TraceId;

    private RequestInfo(String sourceIp, String atlPath, String B3TraceId) {
        this.sourceIp = sourceIp;
        this.atlPath = atlPath;
        this.B3TraceId = B3TraceId;
    }

    public String getSourceIp() {
        return sourceIp;
    }

    public String getAtlPath() {
        return atlPath;
    }

    public String getB3TraceId() { return B3TraceId; }

    public static RequestInfo fromRequest(HttpServletRequest request) {
        return new RequestInfo(extractSourceIP(request), extractAtlPath(request), extractB3TraceIdFinal(request));
    }

    static String extractB3TraceIdFinal (HttpServletRequest httpRequest) {
        if (httpRequest == null) {
            return null;
        }

        Object traceId = httpRequest.getAttribute("B3-TraceId");
        if (traceId != null && traceId instanceof String) {  // Bamboo is running a Struts version that will return integers for missing keys sometimes
            return (String)traceId;
        } else {
            return null;
        }
    }

    @VisibleForTesting
    static String extractSourceIP(final HttpServletRequest httpRequest) {
        if (httpRequest == null) {
            return null;
        }

        String ipAddress = httpRequest.getHeader(X_FORWARDED_FOR_FIELD);
        if (ipAddress == null) {
            ipAddress = httpRequest.getRemoteAddr();
        }

        if (ipAddress != null && ipAddress.contains(",")) {
            return ipAddress.split(",")[0].trim();
        }

        return ipAddress;
    }

    @VisibleForTesting
    static String extractAtlPath(final HttpServletRequest httpRequest) {
        if (httpRequest != null) {
            final Cookie[] cookies = httpRequest.getCookies();
            if (cookies != null) {
                for (Cookie cookie : cookies) {
                    if (ATL_PATH_COOKIE_NAME.equals(cookie.getName())) {
                        return cookie.getValue();
                    }
                }
            }
        }
        return null;
    }
}
