package com.atlassian.analytics.client.sender;

import com.atlassian.analytics.client.configuration.AnalyticsConfig;
import com.atlassian.analytics.client.detect.OnDemandDetector;
import com.atlassian.analytics.event.ProcessedEvent;
import com.atlassian.sal.api.lifecycle.LifecycleAware;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;

import java.util.concurrent.atomic.AtomicReference;

public class AtomicEventSender implements DisposableBean, LifecycleAware
{
    private static final Logger log = LoggerFactory.getLogger(AtomicEventSender.class);

    private final AnalyticsConfig analyticsConfig;
    private final OnDemandDetector onDemandDetector;
    private final AtomicReference<EventSender> sender = new AtomicReference<EventSender>();

    public AtomicEventSender(final AnalyticsConfig analyticsConfig, final OnDemandDetector onDemandDetector)
    {
        this.analyticsConfig = analyticsConfig;
        this.onDemandDetector = onDemandDetector;
    }

    @Override
    public void onStart()
    {
        createSender();
    }

    private void createSender()
    {
        if (onDemandDetector.isOnDemand())
        {
            sender.set(EventSender.newInstance(analyticsConfig));
        }
    }

    public void updateSender()
    {
        EventSender newSender = EventSender.newInstance(analyticsConfig);
        log.info("Setting new analytics destination: " + newSender);
        EventSender oldSender = sender.get();
        if (oldSender != null)
        {
            try
            {
                oldSender.destroy();
            }
            catch (Exception e)
            {
                log.error("Error destroying old event sender. Port may not have been closed properly: " + e.getMessage());
            }
        }
        sender.set(newSender);
    }

    public void sendIfValid(ProcessedEvent processedEvent)
    {
        if (isSenderValid())
        {
            sender.get().send(processedEvent);
        }
    }

    private boolean isSenderValid()
    {
        return sender.get() != null;
    }

    @Override
    public void destroy()
    {
        if (isSenderValid())
        {
            sender.getAndSet(null).destroy();
        }
    }

    @Override
    public void onStop()
    {
        // no-op - use this one in the future in place of destroy()
    }
}
