package com.atlassian.analytics.client.report;

import com.atlassian.analytics.event.AnalyticsEvent;

import java.util.Collection;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;


public class EventReporterStore
{
    private final int capacity;
    private final Set<String> ignoredEvents;
    private final Queue<EventReportItem> events;

    // Use a separate count to avoid calculating the count of the events in the Queue, which can be expensive and
    // inaccurate. There are some race conditions here, but nothing worth worrying about given the nature of the checks
    // are just to restrict the event queue size blowing out to an unreasonable size.
    private final AtomicInteger count;

    public EventReporterStore(final int capacity, final Set<String> ignoredEvents)
    {
        this.capacity = capacity;
        this.ignoredEvents = ignoredEvents;
        this.events = new ConcurrentLinkedQueue<EventReportItem>();
        this.count = new AtomicInteger(0);
    }

    public void add(final AnalyticsEvent message)
    {
        add(message, false);
    }

    public void add(final AnalyticsEvent event, final boolean removed)
    {
        if (!ignoredEvents.contains(event.getName()))
        {
            ensureCapacityLimit();
            events.add(new EventReportItem(event.getName(), event.getReceivedTime(), event.getUser(), event.getRequestCorrelationId(), event.getProperties(), removed));
        }
    }

    public Collection<EventReportItem> getEvents()
    {
        return events;
    }

    public void clear()
    {
        events.clear();
        count.set(0);
    }

    /**
     * Keep the event queue under or at the maximum allowable amount.
     * There is a race condition between the comparison and decrement, however there is no chance of the capacity being
     * exceeded.
     */
    private void ensureCapacityLimit()
    {
        count.incrementAndGet();
        if (count.intValue() > capacity)
        {
            count.decrementAndGet();
            events.poll();
        }
    }
}
