package com.atlassian.analytics.client.report;

import com.atlassian.analytics.client.EventPreprocessor;
import com.atlassian.analytics.event.AnalyticsEvent;
import com.atlassian.analytics.event.ProcessedEvent;
import com.atlassian.analytics.event.RawEvent;
import com.google.common.collect.ImmutableSet;

import java.util.Collection;
import java.util.Set;

public class EventReporter
{
    private static final int CAPACITY = 1000;
    private static final Set<String> IGNORED_EVENTS = ImmutableSet.of(
            "reindexissuesstarted", "reindexissuescompleted");

    private final TimeoutChecker timeoutChecker;
    private final EventPreprocessor eventPreprocessor;
    private final EventReporterStore rawStore;
    private final EventReporterStore btfProcessedStore;
    private final EventReporterStore onDemandProcessedStore;
    private boolean capturing;

    public EventReporter(final TimeoutChecker timeoutChecker, final EventPreprocessor eventPreprocessor)
    {
        this.timeoutChecker = timeoutChecker;
        this.eventPreprocessor = eventPreprocessor;
        this.rawStore = new EventReporterStore(CAPACITY, IGNORED_EVENTS);
        this.btfProcessedStore = new EventReporterStore(CAPACITY, IGNORED_EVENTS);
        this.onDemandProcessedStore = new EventReporterStore(CAPACITY, IGNORED_EVENTS);
        this.capturing = false;
    }

    public void addRawEvent(final RawEvent rawEvent)
    {
        addEvent(rawEvent, rawStore, false);
        addProcessedEvent(rawEvent, true);
        addProcessedEvent(rawEvent, false);
    }

    private void addProcessedEvent(final RawEvent rawEvent, final boolean isOnDemand)
    {
        final EventReporterStore processedStore = isOnDemand ? onDemandProcessedStore : btfProcessedStore;

        if (eventPreprocessor.canCollect(rawEvent, isOnDemand))
        {
            ProcessedEvent processedEvent = eventPreprocessor.preprocess(rawEvent, isOnDemand);
            addProcessedEvent(processedEvent, processedStore);
        }
        else
        {
            addRejectedRawEvent(rawEvent, processedStore);
        }
    }

    public void addProcessedEvent(final ProcessedEvent event, final EventReporterStore processedStore)
    {
        addEvent(event, processedStore, false);
    }

    public void addRejectedRawEvent(final RawEvent event, final EventReporterStore processedStore)
    {
        addEvent(event, processedStore, true);
    }

    private void addEvent(final AnalyticsEvent event, final EventReporterStore store, final boolean removed)
    {
        if (capturing)
        {
            // Check the timeout, so that we don't keep capturing events long after anyone has looked at them.
            if (timeoutChecker.isTimeoutExceeded())
            {
                setCapturing(false);
            }
            else
            {
                store.add(event, removed);
            }
        }
    }

    public Collection<EventReportItem> getRawEvents()
    {
        timeoutChecker.actionHasOccurred();
        return rawStore.getEvents();
    }

    public Collection<EventReportItem> getOnDemandProcessedEvents()
    {
        timeoutChecker.actionHasOccurred();
        return onDemandProcessedStore.getEvents();
    }

    public Collection<EventReportItem> getBtfProcessedEvents()
    {
        timeoutChecker.actionHasOccurred();
        return btfProcessedStore.getEvents();
    }

    public void clear()
    {
        timeoutChecker.actionHasOccurred();
        rawStore.clear();
        onDemandProcessedStore.clear();
        btfProcessedStore.clear();
    }

    public boolean isCapturing()
    {
        return capturing;
    }

    public void setCapturing(boolean capturing)
    {
        timeoutChecker.actionHasOccurred();
        this.capturing = capturing;
    }
}
