package com.atlassian.analytics.client.extractor;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.Set;

import com.atlassian.analytics.api.annotations.Analytics;
import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.analytics.client.api.browser.BrowserEvent;

import com.atlassian.analytics.client.serialize.RequestInfo;
import com.atlassian.util.concurrent.Nullable;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PropertyExtractorHelper
{
    private static final Logger log = LoggerFactory.getLogger(PropertyExtractorHelper.class);
    protected final Iterable<PropertyContributor> propertyDecorators;
    protected final Set<String> excludeProperties;

    public PropertyExtractorHelper(final Set<String> excludeProperties, PropertyContributor... propertyContributors)
    {
        this.excludeProperties = excludeProperties;
        this.propertyDecorators = Arrays.asList(propertyContributors);
    }

    public Map<String, Object> extractProperty(String name, Object value)
    {
        ImmutableMap.Builder<String, Object> result = ImmutableMap.builder();
        if (excludeProperties.contains(name) || value == null)
            return Collections.emptyMap();
        if (value instanceof String || value instanceof Number || value instanceof Boolean)
            result.put(name, value);
        else if (value instanceof Character || value instanceof Enum)
            result.put(name, value.toString());
        else if (value instanceof Date)
            result.put(name, formatDate((Date) value));

        putNonCoreJavaTypes(result, name, value);

        if (value instanceof Collection)
        {
            Collection<?> collection = (Collection<?>) value;
            result.put(name + ".size", String.valueOf(collection.size()));
            int index = 0;
            for (Object o : collection)
            {
                result.putAll(extractProperty(name + "[" + index++ + "]", o));
            }
        }
        if (value instanceof Map)
        {
            Map<?, ?> map = (Map<?, ?>) value;
            result.put(name + ".size", String.valueOf(map.size()));
            for (Map.Entry<?, ?> entry : map.entrySet())
            {
                result.putAll(extractProperty(name + "." + entry.getKey(), entry.getValue()));
            }
        }
        return result.build();
    }

    /**
     * Put types in addition to the core Java types.
     */
    protected void putNonCoreJavaTypes(ImmutableMap.Builder<String, Object> result, String name, Object value)
    {
        for (PropertyContributor propertyContributor : propertyDecorators)
        {
            propertyContributor.contribute(result, name, value);
        }

    }

    public boolean isExcluded(String name)
    {
        return excludeProperties.contains(name);
    }

    /**
     * Produce a 'sub product' identifier that can help identify exactly
     * where this event comes from.
     * <p>
     * In some cases event names overlap, for example usercreated from
     * embedded Crowd and the JIRA user event. In these cases we need to
     * include some mechanism to disambiguate them.
     * <p>
     * This method tries to strike a balance between specificity,
     * readability and size. For this reason we take the package name
     * then:
     * - If it's a BrowserEvent make it just 'browser'
     * - Strip any com?.atlassian?.(product e.g 'jira')? at the start
     * - Remove the words 'event(s)' or 'plugin(s)' from any components
     *   of the package
     * <p>
     * Some examples:
     * - com.atlassian.confluence.plugins.macros.dashboard.recentupdates.events.DashboardRecentlyUpdatedViewEvent =
     *   macros.dashboard.recentupdates
     * - com.atlassian.confluence.event.events.content.mail.notification.SpaceNotificationRemovedEvent =
     *   content.mail.notification
     * - com.atlassian.crowd.event.monitor.poller.PollingFinishedEvent =
     *   crowd.monitor.poller
     * - com.fakevendor.events.UserCreatedEvent = fakevendor
     * - com.atlassian.jira.event.config.ApplicationPropertyChangeEvent = config
     * <p>
     * N.B. Client side events will all show 'browser' because they
     * are processed by the same endpoint in this plugin (and use a
     * com.atlassian.analytics.client.api.browser.BrowserEvent). There's nothing we
     * can do really, we could try to introspect the Javascript stack to
     * see which .js file triggered the event (by overriding the push
     * prototype on the AJS.EventQueue array), but this is very problematic
     * cross browser (and not possible at all in IE less than 10)
     */
    public String extractSubProduct(Object event, String product)
    {
        if (event instanceof BrowserEvent)
            return "browser";

        String eventPackage = event.getClass().getPackage().getName().toLowerCase();
        StringBuilder subProduct = new StringBuilder();
        int wordNo = 0;
        int wordsOut = 0;
        boolean strippingPrefix = true;

        for (String word : eventPackage.split("\\."))
        {
            boolean exclude = false;

            if (wordNo < 3 && strippingPrefix)
            {
                if ((wordNo == 0 && word.equals("com")) ||
                    (wordNo == 1 && word.equals("atlassian")) ||
                    (wordNo == 2 && word.equals(product)))
                    exclude = true;
                else
                    strippingPrefix = false;
            }

            if (!exclude)
            {
                if (word.equals("event")  || word.equals("events") ||
                    word.equals("plugin") || word.equals("plugins"))
                    exclude = true;
            }

            if (!exclude)
            {
                if (wordsOut > 0)
                    subProduct.append('.');
                subProduct.append(word);
                wordsOut++;
            }

            wordNo++;
        }

        return subProduct.toString();
    }

    public String extractName(Object event)
    {
        Class<?> clazz = event.getClass();

        final String nameFromMethod = extractNameFromMethodAnnotation(event, clazz);
        if (null != nameFromMethod)
        {
            return nameFromMethod;
        }

        final String nameFromClass = extractNameFromClassAnnotation(clazz);
        if (null != nameFromClass)
        {
            return nameFromClass;
        }

        // still support the deprecated @Analytics annotation until it's removed
        final String nameFromClassAnalytics = extractNameFromClassAnnotationAnalytics(clazz);
        if (null != nameFromClassAnalytics)
        {
            return nameFromClassAnalytics;
        }

        return extractNameFromClassName(clazz);
    }

    @Nullable
    public String extractRequestCorrelationId(RequestInfo requestInfo)
    {
        return requestInfo.getB3TraceId();
    }

    private String extractNameFromMethodAnnotation(Object event, Class<?> clazz)
    {
        Method[] methods = clazz.getMethods();
        Collection<Method> annotatedMethods = Collections2.filter(Arrays.asList(methods), new Predicate<Method>()
        {
            public boolean apply(Method method)
            {
                try
                {
                    return null != method.getAnnotation(EventName.class);
                }
                catch (NoClassDefFoundError error)
                {
                    // This just means the product is using an old version of the analytics-api JAR and doesn't have this annotation
                    return false;
                }
            }
        });

        int numOfMethods = annotatedMethods.size();
        if (numOfMethods > 0)
        {
            if (numOfMethods > 1)
            {
                log.warn("More than one @EventName annotated methods found in class " + clazz.getName());
            }
            Method method = annotatedMethods.iterator().next();
            try
            {
                Object result = method.invoke(event);
                return result == null ? null : String.valueOf(result);
            }
            catch (IllegalAccessException e)
            {
                log.error("Failed to execute " + clazz.getName() + "." + method.getName() + " to calculate event name: " + e.getMessage(), e);
                return null;
            }
            catch (InvocationTargetException e)
            {
                log.error("Failed to execute " + clazz.getName() + "." + method.getName() + " to calculate event name: " + e.getMessage(), e);
                return null;
            }
        }

        return null;
    }

    private String extractNameFromClassAnnotation(Class<?> clazz)
    {
        try
        {
            final EventName eventName = clazz.getAnnotation(EventName.class);
            if (null != eventName && null != eventName.value() && !"".equals(eventName.value()))
            {
                return eventName.value();
            }
            return null;
        }
        catch (NoClassDefFoundError error)
        {
            // This just means the product is using an old version of the analytics-api JAR and doesn't have this annotation
            return null;
        }
    }

    private String extractNameFromClassAnnotationAnalytics(Class<?> clazz)
    {
        final Analytics analytics = clazz.getAnnotation(Analytics.class);
        if (null != analytics && null != analytics.value() && !"".equals(analytics.value()))
            return analytics.value();
        return null;
    }

    private String extractNameFromClassName(Class<?> clazz)
    {
        String result = clazz.getSimpleName().toLowerCase();
        if (result.endsWith("event"))
            result = result.substring(0, result.length() - 5);

        return result;
    }

    private String formatDate(Date date)
    {
        return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(date);
    }
}
