package com.atlassian.analytics.client.eventfilter.whitelist;

import org.codehaus.jackson.annotate.JsonIgnoreProperties;
import org.codehaus.jackson.annotate.JsonProperty;

import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang.StringUtils;

/**
 * Searches through the whitelists to find whitelisted events.
 */
public class WhitelistSearcher
{
    private final WhitelistFilter whitelistFilter;

    public WhitelistSearcher(final WhitelistFilter whitelistFilter)
    {
        this.whitelistFilter = whitelistFilter;
    }

    public List<SearchResultItem> search(final String query, final String whitelistId, final int maxResults)
    {
        final String lowercaseQuery = (query == null ? "" : query).toLowerCase();
        final List<SearchResultItem> result = new ArrayList<SearchResultItem>();

        search(lowercaseQuery, whitelistId, whitelistFilter.getGlobalWhitelist(), result, maxResults);

        for (final Whitelist whitelist : whitelistFilter.getPluginWhitelists())
        {
            if (result.size() >= maxResults)
            {
                break;
            }

            search(lowercaseQuery, whitelistId, whitelist, result, maxResults);
        }

        return result;
    }

    private void search(final String lowercaseQuery, final String whitelistId, final Whitelist whitelist, final List<SearchResultItem> result, final int maxResults)
    {
        if (StringUtils.isBlank(whitelistId) || whitelist.getWhitelistId().equals(whitelistId))
        {
            for (final String eventName : whitelist.getFilteredEventAttributesMap().keySet())
            {
                if (result.size() >= maxResults)
                {
                    break;
                }

                if (eventName.toLowerCase().contains(lowercaseQuery))
                {
                    result.add(new SearchResultItem(whitelist.getWhitelistId(), eventName, whitelist.getFilteredEventAttributesMap().get(eventName)));
                }
            }
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class SearchResultItem
    {
        @JsonProperty
        private final String whitelistId;

        @JsonProperty
        private final String eventName;

        @JsonProperty
        private final FilteredEventAttributes filteredEventAttributes;

        public SearchResultItem(final String whitelistId, final String eventName, final FilteredEventAttributes filteredEventAttributes)
        {
            this.whitelistId = whitelistId;
            this.eventName = eventName;
            this.filteredEventAttributes = filteredEventAttributes;
        }

        public String getWhitelistId()
        {
            return whitelistId;
        }

        public String getEventName()
        {
            return eventName;
        }

        public FilteredEventAttributes getFilteredEventAttributes()
        {
            return filteredEventAttributes;
        }
    }
}
