package com.atlassian.analytics.client.eventfilter.whitelist;

import com.google.common.collect.ImmutableList;
import com.google.common.io.Closeables;
import org.codehaus.jackson.JsonFactory;
import org.codehaus.jackson.JsonParser;
import org.codehaus.jackson.map.MappingJsonFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.type.TypeReference;
import org.springframework.core.io.Resource;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Simple JSON plugin whitelist reader. Format should be a list of event names, with a list under each containing
 * attribute names. For example:
 * {
 *     "event.name": [
 *          "attributeName1",
 *          "attributeName2"
 *     ],
 *     "event.name2": [],
 *     "event.name3": [
 *          "attributeName3"
 *     ],
 *     "event.name4": [
 *          { "attributeName4": ["allowedValue1", "allowedValue2", "allowedValue3"] }
 *     ]
 * }
 *
 * Attributes will be added to the allowed word dictionary filter list, unless they are "username"-type attributes -
 * those will be automagically added to the hash filter list.
 */
public class PluginWhitelistReader
{
    private static final Logger log = LoggerFactory.getLogger(PluginWhitelistReader.class);

    private static final List<String> HASH_ATTRIBUTE_NAMES = ImmutableList.of("user", "username", "user.name");

    public Map<String, FilteredEventAttributes> read(final InputStream inputStream) throws IOException
    {
        final JsonFactory jf = new MappingJsonFactory();
        jf.enable(JsonParser.Feature.ALLOW_COMMENTS);
        final ObjectMapper mapper = new ObjectMapper(jf);
        // Read simple whitelisted attributes and map them to the default whitelist format
        return mapFilteredAttributes(mapper.<Map<String, List<Object>>>readValue(inputStream, new TypeReference<Map<String, List<Object>>>() {}));
    }

    public Map<String, FilteredEventAttributes> read(Resource resource)
    {
        InputStream inputStream = null;
        try
        {
            inputStream = resource.getInputStream();
            return read(inputStream);
        }
        catch (IOException e)
        {
            log.error("Failed to read the JSON whitelist with error message: {}", e.getMessage());
        }
        finally
        {
            if (inputStream != null)
            {
                Closeables.closeQuietly(inputStream);
            }
        }
        return null;
    }

    private Map<String, FilteredEventAttributes> mapFilteredAttributes(Map<String, List<Object>> attributes)
    {
        Map<String, FilteredEventAttributes> filteredEventAttributesMap = new HashMap<String, FilteredEventAttributes>();

        for (Map.Entry<String, List<Object>> eventAttributesMap : attributes.entrySet())
        {
            final String eventName = eventAttributesMap.getKey();
            FilteredEventAttributes filteredEventAttributes = filteredEventAttributesMap.get(eventName);
            if (filteredEventAttributes == null)
            {
                filteredEventAttributes = new FilteredEventAttributes();
                filteredEventAttributesMap.put(eventName, filteredEventAttributes);
            }
            setFilteredAttributes(filteredEventAttributes, eventAttributesMap.getValue());
        }

        return filteredEventAttributesMap;
    }

    private void setFilteredAttributes(final FilteredEventAttributes filteredEventAttributes, final List<Object> attributeValues)
    {
        List<String> hashAttributes = new ArrayList<String>();
        List<String> dictionaryAttributes = new ArrayList<String>();
        Map<String, List<String>> allowedPropertyValues = new HashMap<String, List<String>>();

        for (Object value : attributeValues)
        {
            // A map will contain the property name and a list of allowed values for that property
            if (value instanceof Map)
            {
                @SuppressWarnings("unchecked")
                Map<String, List<String>> allowedValues = (Map<String, List<String>>) value;
                for (Map.Entry<String, List<String>> allowedValueList : allowedValues.entrySet())
                {
                    final String propertyName = allowedValueList.getKey();
                    addPropertyToFilterList(propertyName, hashAttributes, dictionaryAttributes);
                    allowedPropertyValues.put(propertyName, allowedValueList.getValue());
                }
            }
            // A string will only contain the property name with no additional information
            else if (value instanceof String)
            {
                addPropertyToFilterList((String) value, hashAttributes, dictionaryAttributes);
            }
        }
        attributeValues.removeAll(hashAttributes);

        if (!hashAttributes.isEmpty())
        {
            filteredEventAttributes.setHashedAttributes(hashAttributes);
        }
        if (!attributeValues.isEmpty())
        {
            filteredEventAttributes.setDictionaryFilteredAttributes(dictionaryAttributes);
        }
        if (!allowedPropertyValues.isEmpty())
        {
            filteredEventAttributes.setAllowedPropertyValues(allowedPropertyValues);
        }
    }

    private void addPropertyToFilterList(final String propertyName, final List<String> hashAttributes, final List<String> dictionaryAttributes)
    {
        if (HASH_ATTRIBUTE_NAMES.contains(propertyName))
        {
            hashAttributes.add(propertyName);
        }
        else
        {
            dictionaryAttributes.add(propertyName);
        }
    }
}
