package com.atlassian.analytics.client.eventfilter.whitelist;

import org.apache.commons.lang.StringUtils;
import org.codehaus.jackson.annotate.JsonIgnoreProperties;
import org.codehaus.jackson.annotate.JsonProperty;

import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Filtered event attributes deserialized from JSON format.
 *
 *  - Whitelisted attributes will be allowed through untouched.
 *  - Hashed attributes will be MD5 hashed (with server ID salt)
 *  - Dictionary filtered atributes will remove words not in the list of allowed words
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class FilteredEventAttributes
{
    private static final String NUMERIC_IDENTIFIER = "*";
    private static final Pattern numericPattern = Pattern.compile("(\\d+)");

    @JsonProperty
    private List<String> whitelistedAttributes;

    @JsonProperty
    private List<String> hashedAttributes;

    @JsonProperty
    private List<String> dictionaryFilteredAttributes;

    private Map<String, List<String>> allowedPropertyValues;

    public FilteredEventAttributes() {}

    public void setWhitelistedAttributes(final List<String> whitelistedAttributes)
    {
        this.whitelistedAttributes = whitelistedAttributes;
    }

    public void setHashedAttributes(final List<String> hashedAttributes)
    {
        this.hashedAttributes = hashedAttributes;
    }

    public void setDictionaryFilteredAttributes(final List<String> dictionaryFilteredAttributes)
    {
        this.dictionaryFilteredAttributes = dictionaryFilteredAttributes;
    }

    public void setAllowedPropertyValues(Map<String, List<String>> allowedPropertyValues)
    {
        this.allowedPropertyValues = allowedPropertyValues;
    }

    public boolean hasAllowedAttributeValue(String attributeName, String attributeValue)
    {
        return allowedPropertyValues != null && listContainsAttribute(allowedPropertyValues.get(attributeName), attributeValue);
    }

    public boolean hasWhitelistedAttribute(String attributeName)
    {
        return listContainsAttribute(whitelistedAttributes, attributeName);
    }

    public boolean hasHashedAttribute(String attributeName)
    {
        return listContainsAttribute(hashedAttributes, attributeName);
    }

    public boolean hasDictionaryFilteredAttribute(String attributeName)
    {
        return listContainsAttribute(dictionaryFilteredAttributes, attributeName);
    }

    private boolean listContainsAttribute(final List<String> filterList, final String attributeValue)
    {
        return filterList != null &&
                (filterList.contains(attributeValue) || filterList.contains(getResolvedAttributeName(attributeValue)));
    }

    private static String getResolvedAttributeName(String attributeValue)
    {
        Matcher matcher = numericPattern.matcher(attributeValue);
        while (matcher.find())
        {
            String arrayPattern = matcher.group();
            attributeValue = StringUtils.replaceOnce(attributeValue, arrayPattern, NUMERIC_IDENTIFIER);
        }
        return attributeValue;
    }

    List<String> getWhitelistedAttributes()
    {
        return whitelistedAttributes;
    }

    List<String> getHashedAttributes()
    {
        return hashedAttributes;
    }

    List<String> getDictionaryFilteredAttributes()
    {
        return dictionaryFilteredAttributes;
    }

    Map<String, List<String>> getAllowedPropertyValues()
    {
        return allowedPropertyValues;
    }
}
