package com.atlassian.analytics.client.eventfilter;

import com.atlassian.analytics.client.eventfilter.parser.SimpleListParser;
import com.atlassian.analytics.client.eventfilter.reader.FilterListReader;
import com.atlassian.analytics.client.eventfilter.reader.LocalListReader;
import com.atlassian.analytics.client.eventfilter.reader.RemoteListReader;
import com.atlassian.analytics.client.properties.AnalyticsPropertyService;
import com.atlassian.analytics.event.RawEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * Filter that checks whether events have been blacklisted or not. Blacklisted events should not be sent at all.
 */
public class BlacklistFilter
{
    private static final Logger LOG = LoggerFactory.getLogger(BlacklistFilter.class);

    private final AnalyticsPropertyService applicationProperties;
    private Set<String> blacklistEventNames;
    private Set<Pattern> blacklistEventNamePatterns;

    public BlacklistFilter(AnalyticsPropertyService applicationProperties)
    {
        this.applicationProperties = applicationProperties;
        readBlacklist(new LocalListReader());
    }

    private String getProductName()
    {
        return applicationProperties.getDisplayName().toLowerCase();
    }

    public static String getListName(String appName)
    {
        return "blacklist_" + appName + ".txt";
    }

    public void readRemoteList()
    {
        try
        {
            readBlacklist(new RemoteListReader());
        }
        catch (Exception e)
        {
            LOG.debug("Couldn't read the remote blacklist, keeping the local blacklist for now - exception message: " + e.getMessage());
        }
    }

    public void readBlacklist(FilterListReader filterListReader)
    {
        final SimpleListParser simpleListParser = new SimpleListParser(filterListReader);
        final Set<String> eventNames = simpleListParser.readSimpleFilterList(getListName(getProductName()));

        blacklistEventNames = new HashSet<String>();
        blacklistEventNamePatterns = new HashSet<Pattern>();

        if (eventNames != null)
        {
            for (String eventName : eventNames)
            {
                if (eventName.contains("*"))
                {
                    final Pattern pattern = Pattern.compile("^" + eventName.replaceAll("\\.", "\\\\.").replaceAll("\\*", "\\.\\*") + "$");
                    blacklistEventNamePatterns.add(pattern);
                }
                else
                {
                    blacklistEventNames.add(eventName);
                }
            }
        }
    }

    public boolean isEventBlacklisted(final RawEvent event)
    {
        final String eventName = event.getName();

        if (blacklistEventNames.contains(eventName))
        {
            return true;
        }

        for (Pattern blacklistNamePattern : blacklistEventNamePatterns)
        {
            if (blacklistNamePattern.matcher(eventName).matches())
            {
                return true;
            }
        }
        return false;
    }
}
