package com.atlassian.analytics.client;

import com.atlassian.analytics.client.configuration.AnalyticsConfig;
import com.atlassian.analytics.client.eventfilter.BlacklistFilter;
import com.atlassian.analytics.client.eventfilter.whitelist.WhitelistFilter;
import com.atlassian.analytics.client.logger.EventAnonymizer;
import com.atlassian.analytics.event.ProcessedEvent;
import com.atlassian.analytics.event.RawEvent;

import java.util.Map;

public class EventPreprocessor
{
    private final AnalyticsConfig analyticsConfig;
    private final BlacklistFilter blacklistFilter;
    private final WhitelistFilter whitelistFilter;
    private final EventAnonymizer eventAnonymizer;

    public EventPreprocessor(final AnalyticsConfig analyticsConfig, final BlacklistFilter blacklistFilter,
                             final WhitelistFilter whitelistFilter, final EventAnonymizer eventAnonymizer)
    {
        this.analyticsConfig = analyticsConfig;
        this.blacklistFilter = blacklistFilter;
        this.whitelistFilter = whitelistFilter;
        this.eventAnonymizer = eventAnonymizer;
    }

    /**
     * Check whether the passed event may be collected.
     *
     * @param event the event to test
     * @return whether the event is allowed to be collected or not
     */
    public boolean canCollect(final RawEvent event, final boolean isOnDemand)
    {
        return analyticsConfig.canCollectAnalytics() &&
                !blacklistFilter.isEventBlacklisted(event) &&
                whitelistFilter.isEventWhitelisted(event, isOnDemand);
    }

    /**
     * Preprocess the event, regardless of whether we actually want to send it or not. To check whether it should be
     * collected, use {@link #canCollect}.
     *
     * @param event event that is processed
     */
    public ProcessedEvent preprocess(final RawEvent event, final boolean isOnDemand)
    {
        final Map<String, Object> whitelistedProperties = whitelistFilter.applyWhitelistToEvent(event.getName(),
                                                                                                event.getProperties(),
                                                                                                isOnDemand);
        return new ProcessedEvent.Builder(event)
                .server(anonymizeServer(event, isOnDemand))
                .user(anonymizeUser(event, isOnDemand))
                .sourceIP(anonymizeSourceIP(event, isOnDemand))
                .atlPath(anonymizeAtlPath(event, isOnDemand))
                .properties(whitelistedProperties)
                .build();
    }

    private String anonymizeServer(final RawEvent event, final boolean isOnDemand)
    {
        return isOnDemand ? event.getServer() : eventAnonymizer.hash(event.getServer());
    }

    private String anonymizeUser(final RawEvent event, final boolean isOnDemand)
    {
        return isOnDemand ? event.getUser() : eventAnonymizer.hash(event.getUser());
    }

    private String anonymizeSourceIP(final RawEvent event, final boolean isOnDemand)
    {
        return isOnDemand ? event.getSourceIP() : eventAnonymizer.hash(event.getSourceIP());
    }

    private String anonymizeAtlPath(final RawEvent event, final boolean isOnDemand)
    {
        return isOnDemand ? event.getAtlPath() : eventAnonymizer.hash(event.getAtlPath());
    }
}
