package com.atlassian.analytics.api.extractor;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Map;

/**
 * Extractor which allows a product to define how properties are provided to this analytics module
 * <p>
 * This interface defines commonly used methods from com.atlassian.analytics.client.extractor.PropertyExtractor
 * which require product specific implementations. Allowing the product to define how information is extracted from
 * an event within its own repository and exporting it via OSGi.
 */
public interface ProductProvidedPropertyExtractor {
    /**
     * Extract properties
     *
     * @return a mapping of all valid properties on the value, prefixed with name + '.'
     * or null if the calling class should handle the extraction
     */
    @Nullable
    default Map<String, Object> extractProperty(@Nonnull String name, @Nonnull Object value) {
        return null;
    }

    /**
     * Extract the user from the event
     *
     * @return the user to be associated with this event or null if the calling class should
     * handle the extraction
     */
    @Nullable
    default String extractUser(@Nonnull Object event, @Nonnull Map<String, Object> properties) {
        return null;
    }

    /**
     * Get the application access for the logged in user associated with this event.
     * For un-logged in user, empty string "" should be returned
     *
     * @return comma separated string representing application access or null if the calling
     * class should handle the extraction
     */
    @Nullable
    default String getApplicationAccess() {
        return null;
    }

    /**
     * Enrich properties
     *
     * @return a map of properties to be added to the event or null if the calling
     * class should handle the extraction
     */
    @Nullable
    default Map<String, Object> enrichProperties(@Nonnull Object event) {
        return null;
    }

    /**
     * Extract the name of this event from the event object
     *
     * @return the name of this event or null if the calling
     * class should handle the extraction
     */
    @Nullable
    default String extractName(@Nonnull Object event) {
        return null;
    }
}
