package com.atlassian.plugins.codegen;

import io.atlassian.fugue.Option;

import static io.atlassian.fugue.Option.none;
import static io.atlassian.fugue.Option.some;
import static java.util.Objects.requireNonNull;

/**
 * Simple wrapper for an artifact version string. This can also be a Maven version property,
 * in which case the version string is used as the default value for that property.
 */
public final class VersionId {
    /**
     * Creates an instance with no property name.
     *
     * @param version the version
     * @return a new instance
     */
    public static VersionId version(String version) {
        return new VersionId(some(version), none(String.class));
    }

    /**
     * Creates an instance with a property name.
     *
     * @param propertyName   the property name
     * @param defaultVersion the default version
     * @return a new instance
     */
    public static VersionId versionProperty(String propertyName, String defaultVersion) {
        return new VersionId(some(defaultVersion), some(propertyName));
    }

    /**
     * Creates an instance for no version at all.
     *
     * @return a new instance
     */
    public static VersionId noVersion() {
        return new VersionId(none(String.class), none(String.class));
    }

    private final Option<String> version;
    private final Option<String> propertyName;

    private VersionId(final Option<String> version, final Option<String> propertyName) {
        this.version = requireNonNull(version, "version");
        this.propertyName = requireNonNull(propertyName, "propertyName");
    }

    public Option<String> getVersion() {
        return version;
    }

    public Option<String> getPropertyName() {
        return propertyName;
    }

    public boolean isDefined() {
        return version.isDefined() || propertyName.isDefined();
    }

    /**
     * Returns the version string, unless a property name was specified, in which case it returns
     * "${propertyName}".  Returns none() if neither was specified.
     */
    public Option<String> getVersionOrPropertyPlaceholder() {
        return propertyPlaceholder().orElse(version);
    }

    @Override
    public String toString() {
        return propertyPlaceholder()
                .map(placeholder -> placeholder + defaultValueSuffix())
                .orElse(version)
                .getOrElse("?");
    }

    private String defaultValueSuffix() {
        return version
                .map(v -> " (" + v + ")")
                .getOrElse("");
    }

    private Option<String> propertyPlaceholder() {
        return propertyName.map(property -> "${" + property + "}");
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof VersionId) {
            VersionId v = (VersionId) other;
            return version.equals(v.version) && propertyName.equals(v.propertyName);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return toString().hashCode();
    }
}
