package com.atlassian.adf.util;

import com.atlassian.annotations.Internal;

/**
 * From {@code editor-wikimarkup-transformer/src/char.ts} originally, but there is a lot more in here, now.
 */
@Internal
@SuppressWarnings("UnnecessaryUnicodeEscape")
public abstract class Char {
    private Char() {
        // static-only
    }

    private static final char[] HEX = {
            '0', '1', '2', '3',
            '4', '5', '6', '7',
            '8', '9', 'A', 'B',
            'C', 'D', 'E', 'F'
    };

    /**
     * An <a href="https://en.wikipedia.org/wiki/Dash#En_dash">en dash</a>, {@value}.
     */
    public static final String EN_DASH = "\u2013";

    /**
     * An <a href="https://en.wikipedia.org/wiki/Dash#Em_dash">em dash</a>, {@value}.
     */
    public static final String EM_DASH = "\u2014";

    /**
     * A <a href="https://en.wikipedia.org/wiki/Non-breaking_space">non-breaking space</a>, {@value}.
     */
    public static final String NBSP = "\u00A0";

    /**
     * A paperclip emoji, {@value}.
     */
    public static final String PAPERCLIP = "\uD83D\uDCCE";

    /**
     * A <a href="https://en.wikipedia.org/wiki/Word_joiner">word-joiner</a>, which has no visual
     * representation.
     */
    public static final String WJ = "\u2060";


    /**
     * Checks for a simple ASCII alphabetic character, meaning an uppercase unmodified English-language
     * letter {@code 'A'} through {@code 'Z'}, or the same {@code 'a'} through {@code 'z'} range in lowercase.
     *
     * @param c the character to consider
     * @return {@code true} if the character is a simple ASCII alphanumeric; {@code false} if it is anything else
     */
    public static boolean isAlpha(char c) {
        return isLower(c) || isUpper(c);
    }

    /**
     * Checks for a simple ASCII alphanumeric character, meaning one of the decimal digit {@code '0'} through
     * {@code '9'}, an uppercase unmodified English-language letter {@code 'A'} through {@code 'Z'}, or the
     * same {@code 'a'} through {@code 'z'} range in lowercase.
     *
     * @param c the character to consider
     * @return {@code true} if the character is a simple ASCII alphanumeric; {@code false} if it is anything else
     */
    public static boolean isAlnum(char c) {
        return isAlpha(c) || isDigit(c);
    }

    /**
     * Checks for a simple ASCII hexadecimal digit, meaning a decimal digit {@code '0'} through {@code '9'}
     * or one of the letters {@code 'A'} through {@code 'F'} (allowing both uppercase or lowercase).
     *
     * @param c the character to consider
     * @return {@code true} if the character is a simple ASCII hex digit; {@code false} if it is anything else
     */
    public static boolean isHexit(char c) {
        return isDigit(c)
                || (c >= 'A' && c <= 'F')
                || (c >= 'a' && c <= 'f');
    }

    /**
     * Checks for a simple ASCII digit, meaning a decimal digit {@code '0'} through {@code '9'}.
     *
     * @param c the character to consider
     * @return {@code true} if the character is a simple ASCII digit; {@code false} if it is anything else
     */
    public static boolean isDigit(char c) {
        return c >= '0' && c <= '9';
    }

    /**
     * Checks for a simple ASCII uppercase letter, meaning {@code 'A'} through {@code 'Z'}.
     *
     * @param c the character to consider
     * @return {@code true} if the character is a simple ASCII uppercase letter; {@code false} if it is anything else
     */
    public static boolean isUpper(char c) {
        return c >= 'A' && c <= 'Z';
    }

    /**
     * Checks for a simple ASCII lowercase letter, meaning {@code 'a'} through {@code 'z'}.
     *
     * @param c the character to consider
     * @return {@code true} if the character is a simple ASCII lowercase letter; {@code false} if it is anything else
     */
    public static boolean isLower(char c) {
        return c >= 'a' && c <= 'z';
    }

    /**
     * Checks for a simple ASCII whitespace character as defined by the {@code \s} character class in
     * regular expression patterns.
     *
     * @param c the character to consider
     * @return true if the character matches the character class {@code \s}, which is equivalent to
     * {@code [ \t\n\x0B\f\r]}.
     */
    public static boolean isSpace(char c) {
        switch (c) {
            case '\t':
            case '\n':
            case '\u000B':
            case '\f':
            case '\r':
            case ' ':
                return true;
            default:
                return false;
        }
    }

    /**
     * Checks whether the given character is a line terminator in the sense of what matches the {@code .} character
     * in a non-multiline regex.
     *
     * @param c the character to consider
     * @return true if the character matches the character class, which includes any of the characters
     * {@code U+000A}, {@code U+000D}, {@code U+0085}, {@code U+2028}, or {@code U+2029}.
     */
    public static boolean isLineTerminator(char c) {
        switch (c) {
            case '\n':
            case '\r':
            case '\u0085':
            case '\u2028':
            case '\u2029':
                return true;
            default:
                return false;
        }
    }

    /**
     * Returns the hexadecimal digit for the given value, which must be in the range {@code [0, 15]}.
     * For the values in the range {@code [10, 15]}, uppercase letters are used.
     *
     * @param hexit the value to represent as a hex digit
     * @return the character that represents it, using uppercase for any letters
     * @throws IndexOutOfBoundsException if {@code hexit} is out of the accepted range
     */
    public static char hex(int hexit) {
        return HEX[hexit];
    }
}