package com.atlassian.adf.model.node.type;

import com.atlassian.adf.model.mark.Mark;
import com.atlassian.adf.model.node.Node;

import java.util.Collection;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

/**
 * Indicates that this node can use a {@code marks} element to alter its presentation.
 * <p>
 * The use of marks is often restricted depending on the surrounding context. For example,
 * the {@code text} nodes within a {@code codeBlock} are not allowed to use marks at all.
 *
 * @param <M> the type of marks that this node is able to use
 */
public interface Marked<N extends Node & Marked<N, M>, M extends Mark> extends Node {
    @Override
    N copy();

    /**
     * Returns the type of marks that may be applied to this node.
     *
     * @return the type of marks that may be applied to this node, which is the concrete type for the generic
     * type {@code <M>} on this interface.
     */
    Class<M> markClass();

    /**
     * Returns a list of the marks currently assigned to this node.
     *
     * @return a list of the marks currently assigned to this node.
     */
    Collection<M> marks();

    /**
     * Returns a set of all the {@link Mark.Type mark types} that have been assigned to this node.
     *
     * @return a set of all the {@link Mark.Type mark types} that have been assigned to this node.
     */
    Set<String> markTypes();

    /**
     * Returns a stream of the marks that are currently assigned to this node and match the given type.
     * <p>
     * Since only one mark of each type is permitted to be assigned to a single node, the stream
     * will not contain multiple elements when {@code markClass} is a concrete mark class.
     *
     * @param markClass the mark types to be included in the stream
     * @param <T>       the inferred mark class
     * @return the stream of matching marks
     */
    <T extends M> Stream<? extends T> marks(Class<T> markClass);

    /**
     * Returns the existing mark of the given type, if it exists.
     *
     * @param type the mark type to look up
     * @return the existing mark, or {@code empty} if no such mark does exist
     */
    Optional<M> mark(String type);

    /**
     * Adds the given mark to this marked node.
     *
     * @param mark the mark to be added
     * @return {@code this}
     * @throws IllegalArgumentException if the mark cannot be added because it conflicts with an existing mark,
     *                                  such as trying to add a second {@code link} mark to the same piece of text.
     * @throws IllegalStateException    if the mark cannot be added because marks are disabled or otherwise
     *                                  restricted in the node's current context.
     */
    N mark(M mark);
}
