package com.atlassian.adf.model.node.type;

import com.atlassian.adf.model.Element;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.node.ListItem;
import com.atlassian.adf.model.node.MediaSingle;
import com.atlassian.adf.model.node.Node;

import java.util.Collection;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Describes the behaviour common to all nodes that act as holders of other content.
 * <p>
 * For the most part, this interface is implemented by all node types that have a {@code "content"} section,
 * with the {@code mediaSingle} node type as the notable exception.
 *
 * @param <C> this content node's type
 * @param <N> the node type that it holds
 */
public interface ContentNode<C extends ContentNode<C, N>, N extends Node>
        extends Node {

    @Override
    C copy();

    /**
     * @return {@code true} if this content node is completely empty, which for many such nodes is not valid
     */
    boolean isEmpty();

    /**
     * Returns the existing contents of this content node.
     * <p>
     * Notes:
     * <ul>
     *     <li>The returned list cannot be modified directly.</li>
     *     <li>Whether it is a live view of the node's contents or a copy is left unspecified. Callers are
     *     advised to {@link List#copyOf(Collection) make an explicit copy of the list} if this matters,
     *     such as when the node's contents will be modified while iterating over this list.</li>
     *     <li>The nodes within the list will generally be mutable. The caller should make an
     *     {@link Element#copy() explicit copy of the node} before modifying it if the original is
     *     not meant to be affected.</li>
     * </ul>
     *
     * @return the existing contents of this content node.
     */
    List<N> content();

    /**
     * Adds the given node as content within this containing node.
     * <p>
     * The content is processed in the same way as if it were the single value provided to
     * {@link #content(Node[])}.
     *
     * @param content the content to add
     * @return {@code this}
     * @see #content(Node[])
     */
    C content(N content);

    /**
     * Adds the given nodes as content within this containing node.
     * <p>
     * Each content item is validated as acceptable content before it is added. If validation is successful,
     * then it is added to this node's content before the next node is validated, so the validation method
     * is permitted to rely on the {@link #content} value being up-to-date at all times.
     * <p>
     * If validation fails for a particular content item, then those particular content items are discarded,
     * but any content nodes that follow will still be attempted.
     *
     * @param content the content to add
     * @return {@code this}
     * @throws AdfException for the first content item to fail validation, with additional failures
     *                      included as {@link Throwable#addSuppressed(Throwable) suppressed} exceptions.
     *                      Note that the reported index is that of the array index on this method, not
     *                      its position in the node's content.
     */
    // Arrays.asList(x).iterator() is always type-safe. Normally, we should just make the method `final` and
    // add the `@SafeVarargs` annotation to it, but that isn't allowed on an `interface` method. This method is
    // really useful, so we are suppressing the warnings instead. This is justified because:
    //  * The implementation in `AbstractContentNode` is safe and `final`.
    //  * There shouldn't really be any need for anyone to implement this interface directly.
    //  * If they did, they would still get a varargs warning of their very own to contend with, anyway.
    @SuppressWarnings({"varargs", "unchecked"})
    C content(N... content);

    /**
     * Adds the given nodes as content within this containing node.
     * <p>
     * The content is processed just as if its elements were passed to {@link #content(Node[])}.
     *
     * @param content the content to add
     * @return {@code this}
     * @see #content(Node[])
     */
    C content(Iterable<? extends N> content);

    /**
     * Adds the given nodes as content within this containing node.
     * <p>
     * Each content item is validated as acceptable content before it is added. If validation is successful,
     * then it is added to this node's content before the next node is validated, so the validation method
     * is permitted to rely on the {@link #content} value being up-to-date at all times.
     * <p>
     * If validation fails for a particular content item, then those particular content items are discarded,
     * but any content nodes that follow will still be attempted.
     *
     * @param content the content to add
     * @return {@code this}
     * @throws AdfException for the first content item to fail validation, with additional failures
     *                      included as {@link Throwable#addSuppressed(Throwable) suppressed} exceptions.
     *                      Note that the reported index is that of the stream provided to this method.
     */
    C content(Stream<? extends N> content);

    /**
     * Produces a stream consisting of the in-order traversal of this node and all of its contents,
     * where any nested {@code ContentNode}s are also expanded.
     * <h2>Example</h2>
     * <p>If a {@code Doc} is constructed from this input:</p>
     * <pre>{@code
     *   {
     *     "type": "paragraph",
     *     "content": [
     *       {
     *         "type": "text",
     *         "text": "Hello "
     *       },
     *       {
     *         "type": "text",
     *         "text": "world",
     *         "marks": [
     *           {
     *             "type": "textColor",
     *             "attrs": {
     *               "color": "#00ff00"
     *             }
     *           }
     *         ]
     *       },
     *       {
     *         "type": "text",
     *         "text": "!"
     *       }
     *     ]
     *   }
     * }</pre>
     * <p>then calling {@code doc.allNodes()} would produce the following stream of nodes:</p>
     * <ul>
     *     <li><p>The {@code doc} node itself</p></li>
     *     <li><p>The {@code paragraph} node</p></li>
     *     <li><p>The {@code text} node containing {@code "Hello "}</p></li>
     *     <li><p>The {@code text} node containing {@code "world"}</p></li>
     *     <li><p>The {@code text} node containing {@code "!"}</p></li>
     * </ul>
     * <p>Note that marks are not considered separate nodes; they are part of the containing {@code text} node.</p>
     *
     * @return a stream of this node and the deep expansion of its contents
     */
    Stream<Node> allNodes();

    /**
     * Discards all existing content in this node.
     * <p>
     * Note that for some node types, it is not valid for the node to be left empty and
     * new content must be added to replace it.
     *
     * @return {@code this}
     */
    C clear();

    /**
     * Discards all content that matches the given predicate.
     * <p>
     * <strong>WARNING</strong>: This uses {@link #replaceContent(List)} in an attempt to ensure that the node
     * is left in a valid state. All the same caveats apply.
     *
     * @param predicate test for which content nodes should be removed
     */
    void removeIf(Predicate<? super N> predicate);

    /**
     * Completely replace this content node's existing content items with the new ones provided.
     * If this results in an exception being thrown (such as rejecting the new first content item
     * in a node like {@link ListItem} that has restrictions for it), then the old content is
     * restored before the exception is propagated to the caller.
     * <p>
     * <strong>WARNING</strong>: Although this method tries very hard to ensure that the content node is left
     * in a valid state, it does still allow content to be removed, possibly leaving it entirely empty. Just
     * as with {@link #clear()}, it is up to the caller to follow this up by adding new content as required
     * to establish a valid state for the content node, or the problem may not be detected until the node is
     * {@link #validate() validated} during some future use. Proceed with caution.
     *
     * @param content the new content list for this node
     */
    void replaceContent(List<? extends N> content);

    /**
     * Returns a stream of all descendant nodes of the given type.
     * <p>
     * The search is an in-order traversal of all child nodes.
     *
     * @param nodeClass the node type to include in the stream
     * @param <T>       {@code nodeClass}, as a type inference
     * @return a stream of all descendant nodes of the given type
     */
    default <T extends Node> Stream<T> allNodesOfType(Class<T> nodeClass) {
        return allNodes()
                .filter(nodeClass::isInstance)
                .map(nodeClass::cast);
    }

    /**
     * Returns a list of all decendant nodes of the given type.
     * <p>
     * The list is the result of an in-order traversal of all child nodes.
     *
     * @param nodeClass the node type to include in the list
     * @param <T>       {@code nodeClass}, as a type inference
     * @return a list of all descendant nodes of the given type
     */
    default <T extends Node> List<T> allNodesOfTypeAsList(Class<T> nodeClass) {
        return allNodesOfType(nodeClass).collect(Collectors.toList());
    }

    /**
     * Applies a transformation to the immediate children of this content node.
     * <p>
     * The {@code transformer} has the following characteristics:
     * </p>
     * <ol>
     *     <li>It will be called once for each piece of content currently held by this parent content node.</li>
     *     <li>It may return that same object to keep that piece of content.</li>
     *     <li>It may return same other object that is suitable for the parent content node, and that node
     *     will replace the original value.</li>
     *     <li>It may return {@code null}, in which case that content item will be removed, instead.</li>
     * </ol>
     * <p>
     * <strong>WARNING</strong>: Once all content items have been processed by the transformer, this method
     * uses {@link #replaceContent(List)} to perform the update if the structure of the list has changed
     * (that is, if any content items are removed or replaced). In addition to the warnings on that method,
     * note that since the transformer has already been called on the content items, they may have been
     * modified in ways that impact the validity of the content node even if its content item list is
     * left unchanged. If so, this may not be detected until some time later when the node's validity is
     * rechecked for some other reason. Proceed with caution.
     *
     * @param transformer the transforming function
     */
    void transformContent(Function<? super N, ? extends N> transformer);

    /**
     * Applies a transformation to all descendants of this node that match the given type.
     * <p>
     * The {@code transformer} has the following characteristics:
     * </p>
     * <ol>
     *     <li>It will be called once for each piece of content currently held by this content node or any
     *     of its descendants that matches the given {@code targetNodeClass}.</li>
     *     <li>It may return that same object to keep that node within the content node that contains it.</li>
     *     <li>It may return same other object that is suitable for the parent content node that contained it,
     *     and that new node will replace the original value.</li>
     *     <li>It may return {@code null}, in which case that content item will be removed, instead.</li>
     * </ol>
     * <p>
     * <strong>WARNING</strong>: Once all content items have been processed by the transformer, this method
     * uses {@link #replaceContent(List)} to perform the update if the structure of the list has changed
     * (that is, if any content items are removed or replaced). In addition to the warnings on that method,
     * note that since the transformer has already been called on the content items, they may have been
     * modified in ways that impact the validity of the content node even if its content item list is
     * left unchanged. If so, this may not be detected until some time later when the node's validity is
     * rechecked for some other reason. Proceed with caution.
     * <h2>IMPLEMENTATION NOTE</h2>
     * This method has special support for {@link MediaSingle} nodes, which do not implement the
     * {@code ContentNode} interface even though they contain child nodes. In particular:
     * <ol>
     *     <li>The {@code media} and (if present) {@code caption} nodes are included in the traversal</li>
     *     <li>If the {@code transformer} removes a {@code media} node that is inside of a {@code mediaSingle},
     *         then that {@code mediaSingle} is also removed without visiting its {@code caption}.</li>
     * </ol>
     *
     *
     * @param transformer the transforming function
     */
    <T extends Node> void transformDescendants(
            Class<T> targetNodeClass,
            Function<? super T, ? extends T> transformer
    );

}
