package com.atlassian.adf.model.node.type;

import com.atlassian.adf.model.node.Node;
import com.atlassian.adf.util.Fold2;

import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * Describes features that are common to {@code blockCard} and {@code inlineCard} nodes.
 */
public interface CardNode
        extends Node {

    /**
     * Returns the card's link data, either as a {@code URI} or as a {@code Map} containing a JSON-LD.
     *
     * @return the card's link data, either as a {@code URI} or as a {@code Map} containing a JSON-LD.
     */
    UrlOrData urlOrData();

    /**
     * A sum type providing access to the link data for this card.
     */
    interface UrlOrData extends Fold2<String, Map<String, ?>>  {
        /**
         * Transforms the link data into a desired resulting output form.
         * This method provides the {@code url} value as a raw string without verifying that it is a valid URI.
         *
         * @param ifUrl  the mapping to be applied if the link data is in the URI format.
         * @param ifData the mapping to be applied if the link data is in JSON-LD format.
         * @param <T>    the inferred output data type of the mappings
         * @return the result of applying the appropriate mapping function
         */
        @Override
        <T> T fold(
                Function<? super String, ? extends T> ifUrl,
                Function<? super Map<String, ?>, ? extends T> ifData
        );

        /**
         * Do something with either the URI or JSON-LD data, as appropriate.
         * This method provides the {@code url} value as a raw string without verifying that it is a valid URI.
         *
         * @param ifUrl  what to do with the URI if that is the form the link was provided in.
         * @param ifData what to do with the JSON-LD if that is the form the link was provided in.
         */
        @Override
        void accept(
                Consumer<? super String> ifUrl,
                Consumer<? super Map<String, ?>> ifData
        );

        /**
         * Return the {@code url} property, if that is the form that the link was provided in.
         *
         * @return the {@code url} property, if that is the form that the link was provided in;
         * otherwise {@code empty()}.
         */
        Optional<String> url();

        /**
         * Return the JSON-LD data, if that is the form that the link was provided in.
         *
         * @return the JSON-LD data, if that is the form that the link was provided in; otherwise {@code empty()}.
         */
        Optional<Map<String, ?>> data();
    }
}
