package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.mark.Code;
import com.atlassian.adf.model.mark.Em;
import com.atlassian.adf.model.mark.Link;
import com.atlassian.adf.model.mark.Mark;
import com.atlassian.adf.model.mark.Strike;
import com.atlassian.adf.model.mark.Strong;
import com.atlassian.adf.model.mark.SubSup;
import com.atlassian.adf.model.mark.TextColor;
import com.atlassian.adf.model.mark.Underline;
import com.atlassian.adf.model.mark.type.CodeTextMark;
import com.atlassian.adf.model.mark.type.FormattedTextMark;
import com.atlassian.adf.model.mark.type.TextMark;
import com.atlassian.adf.model.mark.unsupported.UnsupportedTextMark;
import com.atlassian.adf.model.node.type.CaptionContent;
import com.atlassian.adf.model.node.type.ContentNode;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Colors;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.net.URL;
import java.util.Arrays;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import static com.atlassian.adf.model.Element.nonEmpty;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * Holds document text. The HTML equivalent is any bare text to be displayed, such as
 * would be found inside a {@link Paragraph paragraph}. Except where otherwise noted,
 * text nodes can be augmented with any of the {@link Mark mark} elements to enhance
 * how they are displayed.
 * <p>
 * Empty text nodes are not permitted; the non-empty text value must be provided.
 * </p>
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Paragraph#p(InlineContent) p}({@link #text(String) text}("Hello world"))
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "paragraph",
 *     "content": [
 *       {
 *         "type": "text",
 *         "text": "Hello world"
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <p>Hello world</p>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/text/">Node - text</a>
 */
@Documentation(
        state = Documentation.State.INCOMPLETE,
        date = "2023-07-26",
        comment = "documentation around marks is unclear and omits the 'annotation' mark"
)
public class Text
        extends AbstractMarkedNode<Text, TextMark>
        implements CaptionContent, InlineContent {

    static Factory<Text> FACTORY = new Factory<>(Type.TEXT, Text.class, Text::parse);

    private String text;

    private Text(String text) {
        this.text = validateText(requireNonNull(text, "text"));
    }

    private Text(String text, Stream<? extends TextMark> marks) {
        this(text);
        marks.forEach(this::mark);
    }

    /**
     * Create a new text node for the given input string without any marks.
     *
     * @param text the text that the node contains; must not be empty
     * @return the new text node
     * @throws IllegalArgumentException if the text value is empty
     */
    public static Text text(String text) {
        return new Text(text);
    }

    /**
     * Create a stream of text nodes for the given input strings without any marks.
     *
     * @param text the text values that the nodes will contain; none of them can be empty
     * @return the stream of new text nodes
     * @throws IllegalArgumentException if any text value is empty or if the specified marks
     *                                  {@link Mark are not compatible}
     */
    public static Stream<Text> text(String... text) {
        return Arrays.stream(text).map(Text::text);
    }

    /**
     * Create a stream of text nodes for the given input strings without any marks.
     *
     * @param text the text values that the nodes will contain; none of them can be empty
     * @return the stream of new text nodes
     * @throws IllegalArgumentException if any text value is empty or if the specified marks
     *                                  {@link Mark are not compatible}
     */
    public static Stream<Text> text(Iterable<? extends String> text) {
        return StreamSupport.stream(text.spliterator(), false).map(Text::text);
    }

    /**
     * Create a stream of text nodes for the given input strings without any marks.
     *
     * @param text the text values that the nodes will contain; none of them can be empty
     * @return the stream of new text nodes
     * @throws IllegalArgumentException if any text value is empty or if the specified marks
     *                                  {@link Mark are not compatible}
     */
    public static Stream<Text> text(Stream<? extends String> text) {
        return text.map(Text::text);
    }

    /**
     * Create a new text node that also has the specified marks.
     *
     * @param text the text that the node contains; must not be empty
     * @return the new text node
     * @throws IllegalArgumentException if any text value is empty or if the specified marks
     *                                  {@link Mark are not compatible}
     */
    public static Text text(String text, TextMark... marks) {
        return new Text(text, Arrays.stream(marks));
    }

    /**
     * @see #text(String, TextMark[])
     */
    public static Text text(String text, Iterable<? extends TextMark> marks) {
        return new Text(text, StreamSupport.stream(marks.spliterator(), false));
    }

    /**
     * @see #text(String, TextMark[])
     */
    public static Text text(String text, Stream<? extends TextMark> marks) {
        return new Text(text, marks);
    }

    @Override
    public Text copy() {
        return parse(toMap());
    }

    // This should be named "text" for consistency, but in Java, you cannot have both
    // static and non-static methods with the same signature, and text(String) is already
    // used for the factory method, which is more important.
    public Text set(String text) {
        this.text = validateText(text);
        return this;
    }

    public Text append(String text) {
        this.text += text;
        return this;
    }

    /**
     * This converts the text string to uppercase using {@code Locale#ROOT} semantics.
     * Note that this will not be correct for all locales, most notably Turkish, where it will
     * incorrectly map {@code i} to {@code I} when it should be <code>&#x130;</code>
     * (U+0130; LATIN CAPITAL LETTER I WITH DOT ABOVE).
     *
     * @return {@code this}
     */
    public Text uppercase() {
        return set(text.toUpperCase(Locale.ROOT));
    }

    /**
     * This converts the text string to lowercase using {@code Locale#ROOT} semantics.
     * Note that this will not be correct for all locales, most notably Turkish, where it will
     * incorrectly map {@code I} to {@code i} when it should be <code>&#x131;</code>
     * (U+0131; LATIN SMALL LETTER DOTLESS I).
     *
     * @return {@code this}
     */
    public Text lowercase() {
        return set(text.toLowerCase(Locale.ROOT));
    }

    /**
     * Returns the text that is contained within this text node; never {@code null} or empty.
     *
     * @return the text that is contained within this text node; never {@code null} or empty.
     */
    public String text() {
        return text;
    }

    public boolean hasSameMarks(Text other) {
        return marks.equals(other.marks);
    }

    @Override
    public Class<TextMark> markClass() {
        return TextMark.class;
    }

    @Override
    public Text mark(TextMark mark) {
        marks.add(mark);
        applyMarkRestrictions(mark);
        return this;
    }

    private void applyMarkRestrictions(TextMark mark) {
        if (mark instanceof CodeTextMark) {
            if (!(mark instanceof FormattedTextMark)) {
                marks.restrictToInstancesOf(CodeTextMark.class, "Already used a code-only mark");
            }
        } else if (mark instanceof FormattedTextMark) {
            marks.restrictToInstancesOf(FormattedTextMark.class, "Already used a formatted-text-only mark");
        }
    }

    /**
     * Adds the given mark to the node if it is permitted.
     * <p>
     * This is intended for cases where the caller would like to apply a mark, but does not consider it
     * an error if that mark is not allowed and does not want to have to catch an exception for it.
     * A common example might be an attempt to mark text as {@link #strong()}, but not wanting to check
     * first whether this will be permitted or handle the exception if it is not.
     * <p>
     * Some reasons why the mark might be disallowed include:
     * <ol>
     *     <li>A mark of that type is already present.</li>
     *     <li>It would result in a {@link CodeTextMark} and a {@link FormattedTextMark} on the same node.</li>
     *     <li>The text node is already the child of another node type (such as {@code codeBlock}) that only
     *     accepts unmarked text nodes as children.</li>
     * </ol>
     *
     * @param mark the mark to add to the text node, provided it is compatible with the parent node and
     *             any other marks that it already has
     * @return {@code true} if the mark was successfully added; {@code false} if the mark was disallowed
     */
    public boolean markIfAllowed(TextMark mark) {
        if (marks.addIfAllowed(mark).isPresent()) return false;
        applyMarkRestrictions(mark);
        return true;
    }

    /**
     * Add a {@link Code} mark to this text.
     *
     * @return {@code this}
     */
    public Text code() {
        return mark(Code.code());
    }

    /**
     * Add an {@link Em} mark to this text.
     *
     * @return {@code this}
     */
    public Text em() {
        return mark(Em.em());
    }

    /**
     * Add a {@link Link} mark to this text using the given target URL.
     * <p>
     * This convenience method doesn't support setting any of the optional attributes for a link. Construct one
     * manually using the {@link Link} class directly and add it with {@link #mark(TextMark)} if these are needed.
     *
     * @param url the target URL to be used as the {@code href} attribute for this link
     * @return {@code this}
     */
    public Text link(String url) {
        return mark(Link.link(url));
    }

    /**
     * Add a {@link Link} mark to this text using the given target URL.
     * <p>
     * This convenience method doesn't support setting any of the optional attributes for a link. Construct one
     * manually using the {@link Link} class directly and add it with {@link #mark(TextMark)} if these are needed.
     *
     * @param url the target URL to be used as the {@code href} attribute for this link
     * @return {@code this}
     */
    public Text link(URL url) {
        return mark(Link.link(url));
    }

    /**
     * Add a {@link Strike} mark to this text.
     *
     * @return {@code this}
     */
    public Text strike() {
        return mark(Strike.strike());
    }

    /**
     * Add a {@link Strong} mark to this text.
     *
     * @return {@code this}
     */
    public Text strong() {
        return mark(Strong.strong());
    }

    /**
     * Add a {@link SubSup#sub() sub} mark to this text.
     *
     * @return {@code this}
     */
    public Text sub() {
        return mark(SubSup.sub());
    }

    /**
     * Add a {@link SubSup#sup() sup} mark to this text.
     *
     * @return {@code this}
     */
    public Text sup() {
        return mark(SubSup.sup());
    }

    /**
     * Add a {@link TextColor} mark to this text.
     *
     * @param color a color defined in HTML hexadecimal format, such as {@code #daa520}.
     * @return {@code this}
     */
    public Text textColor(String color) {
        return mark(TextColor.textColor(color));
    }

    /**
     * Add a {@link TextColor} mark to this text.
     *
     * @param color a color
     * @return {@code this}
     */
    public Text textColor(java.awt.Color color) {
        return mark(TextColor.textColor(color));
    }

    /**
     * Add a {@link TextColor} mark to this text.
     *
     * @param color one of the standard {@link Colors.Named named colors}
     * @return {@code this}
     */
    public Text textColor(Colors.Named color) {
        return mark(TextColor.textColor(color));
    }

    /**
     * Add an {@link Underline} mark to this text.
     *
     * @return {@code this}
     */
    public Text underline() {
        return mark(Underline.underline());
    }

    @Override
    public String elementType() {
        return Type.TEXT;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.TEXT, text)
                .let(marks::addToMap);
    }

    @Override
    protected boolean markedNodeEquals(Text other) {
        return text.equals(other.text);
    }

    @Override
    protected int markedNodeHashCode() {
        return text.hashCode();
    }

    @Override
    protected void appendMarkedNodeFields(ToStringHelper buf) {
        buf.appendTextField(text);
    }

    @Nullable
    @Override
    protected Factory<TextMark> unsupportedMarkFactory() {
        return UnsupportedTextMark.FACTORY;
    }

    void disableMarks(ContentNode<?, ? super Text> parent) {
        marks.disable(parent.elementType());
    }

    private static Text parse(Map<String, ?> map) {
        checkType(map, Type.TEXT);
        String text = getOrThrow(map, Key.TEXT);
        return text(text).parseMarks(map);
    }

    protected String validateText(String text) {
        return nonEmpty(text, "text");
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        sb.append(text);
    }
}