package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.stream.Stream;

import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * A table cell within a table row.
 * <div style="color: black; background-color: #fffae6; border-radius: 3px; padding: 16px; width: 75%;">
 * \u26A0\uFE0F <strong>WARNING</strong>: Tables are documented as supported on <strong>web</strong> and
 * <strong>desktop</strong> only. <strong>Mobile</strong> rendering support for tables may be restricted
 * or entirely unavailable.</div>
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #td(TableCellContent) td}({@link Paragraph#p(String) p}("Hello world"));
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "tableCell",
 *     "content": [
 *       {
 *         "type": "paragraph",
 *         "content": [
 *           {
 *             "type": "text",
 *             "text": "Hello world"
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <table summary="">
 *     <tr>
 *         <td><p>Hello world</p></td>
 *     </tr>
 * </table>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/tableCell/">Node - tableCell</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
@SuppressWarnings("UnnecessaryUnicodeEscape")
public class TableCell extends AbstractTableCellNode<TableCell> {
    static Factory<TableCell> FACTORY = new Factory<>(Type.TABLE_CELL, TableCell.class, TableCell::parse);

    private TableCell() {
    }

    /**
     * @return a new, empty table cell. At least one content item must be added to make the table cell valid.
     */
    public static TableCell td() {
        return new TableCell();
    }

    /**
     * @return a new table cell with the given string wrapped in a {@code Paragraph} as content
     */
    public static TableCell td(String content) {
        return td().content(content);
    }

    /**
     * @return a new table cell with the given strings wrapped in a {@code Paragraph} as content
     */
    public static TableCell td(String... content) {
        return td().content(content);
    }

    /**
     * @return a new table cell with the given content
     */
    public static TableCell td(TableCellContent content) {
        return td().content(content);
    }

    /**
     * @return a new table cell with the given content
     */
    public static TableCell td(TableCellContent... content) {
        return td().content(content);
    }

    /**
     * @return a new table cell with the given content
     */
    public static TableCell td(Iterable<? extends TableCellContent> content) {
        return td().content(content);
    }

    /**
     * @return a new table cell with the given content
     */
    public static TableCell td(Stream<? extends TableCellContent> content) {
        return td().content(content);
    }

    /**
     * @see #td()
     */
    public static TableCell tableCell() {
        return td();
    }

    /**
     * @see #td(TableCellContent)
     */
    public static TableCell tableCell(TableCellContent content) {
        return td().content(content);
    }

    /**
     * @see #td(TableCellContent[])
     */
    public static TableCell tableCell(TableCellContent... content) {
        return td().content(content);
    }

    /**
     * @see #td(Iterable)
     */
    public static TableCell tableCell(Iterable<? extends TableCellContent> content) {
        return td().content(content);
    }

    /**
     * @see #td(Stream)
     */
    public static TableCell tableCell(Stream<? extends TableCellContent> content) {
        return td().content(content);
    }

    @Override
    public TableCell copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.TABLE_CELL;
    }

    /**
     * Returns {@code false} unconditionally, because this is an ordinary table cell.
     *
     * @return {@code false} unconditionally, because this is an ordinary table cell.
     */
    @Override
    public boolean isHeader() {
        return false;
    }

    private static TableCell parse(Map<String, ?> map) {
        checkType(map, Type.TABLE_CELL);
        return td().parseTableNode(map);
    }
}
