package com.atlassian.adf.model.node;

import com.atlassian.adf.model.ex.node.NodeException;
import com.atlassian.adf.model.node.unsupported.UnsupportedNodeFactory;
import com.atlassian.adf.util.Factory;

import java.util.Map;

import static com.atlassian.adf.model.ex.AdfException.NODE_PREFIX;
import static com.atlassian.adf.model.ex.AdfException.frame;
import static com.atlassian.adf.util.Factory.extractLookupMap;
import static com.atlassian.adf.util.ParserSupport.getTypeOrThrow;

/**
 * Used by the various {@link AbstractContentNode content nodes} to process the other nodes that they contain.
 *
 * @see Doc#parse(Map)
 */
public abstract class NodeParserSupport {
    private static final Map<String, Factory<? extends Node>> NODE_MAP = extractLookupMap(
            Factory::type,
            BlockCard.FACTORY,
            Blockquote.FACTORY,
            BodiedExtension.FACTORY,
            BulletList.FACTORY,
            Caption.FACTORY,
            CodeBlock.FACTORY,
            Date.FACTORY,
            DecisionItem.FACTORY,
            DecisionList.FACTORY,
            Doc.FACTORY,
            EmbedCard.FACTORY,
            Emoji.FACTORY,
            Expand.FACTORY,
            Extension.FACTORY,
            HardBreak.FACTORY,
            Heading.FACTORY,
            InlineCard.FACTORY,
            InlineExtension.FACTORY,
            LayoutColumn.FACTORY,
            LayoutSection.FACTORY,
            ListItem.FACTORY,
            Media.FACTORY,
            MediaGroup.FACTORY,
            MediaInline.FACTORY,
            MediaSingle.FACTORY,
            Mention.FACTORY,
            NestedExpand.FACTORY,
            OrderedList.FACTORY,
            Panel.FACTORY,
            Paragraph.FACTORY,
            Placeholder.FACTORY,
            Rule.FACTORY,
            Status.FACTORY,
            Table.FACTORY,
            TableCell.FACTORY,
            TableHeader.FACTORY,
            TableRow.FACTORY,
            TaskItem.FACTORY,
            TaskList.FACTORY,
            Text.FACTORY
    );

    private NodeParserSupport() {
        // static-only class
    }

    /**
     * Parses an ADF document or an ADF fragment in isolation.
     *
     * @param map the encoded ADF content
     * @return the parsed node, which may be an entire document, but does not have to be
     */
    public static Node fragment(Map<String, ?> map) {
        String type = getTypeOrThrow(map);
        return Node.Type.DOC.equals(type)
                ? Doc.parse(map)
                : getNodeOfType(Node.class, map, "...");
    }

    static <N extends Node> N getNodeOfType(
            Class<N> requiredClass,
            Map<String, ?> map,
            Node parent
    ) {
        return getNodeOfType(requiredClass, map, parent.elementType());
    }

    static <N extends Node> N getNodeOfType(
            Class<N> requiredClass,
            Map<String, ?> map,
            String parentType
    ) {
        String type = getTypeOrThrow(map);
        return frame(NODE_PREFIX + type, () -> {
            Factory<?> factory = NODE_MAP.get(type);
            if (factory == null) {
                return UnsupportedNodeFactory.unsupportedNode(parentType, requiredClass, map);
            }

            if (!requiredClass.isAssignableFrom(factory.typeClass())) {
                throw new NodeException.TypeMismatch(parentType, requiredClass, type);
            }

            return requiredClass.cast(factory.parse(map));
        });
    }
}
