package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.CaptionContent;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttr;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * Represents a user mention.
 * <h2>Implementation Note</h2>
 * Although the documentation implies that only the enumerated {@link AccessLevel} values should be used
 * for that attribute, the reality is that this was never enforced by the schema, and the editor itself
 * frequently specifies it as an empty string.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #mention(String) mention}("ABCDE-ABCDE-ABCDE-ABCDE")
 *         .{@link #text(String) text}("@jsmith")
 *         .{@link #userType(UserType) userType}({@link UserType#APP APP});
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "mention",
 *     "attrs": {
 *       "id": "ABCDE-ABCDE-ABCDE-ABCDE",
 *       "text": "@jsmith",
 *       "userType": "APP"
 *     }
 *   }
 * }</pre>
 */
@Documentation(
        state = Documentation.State.WRONG,
        date = "2023-07-26",
        comment = "the 'accessLevel' attribute is not restricted and an empty string is often used for it"
)
@SuppressWarnings({"UnusedReturnValue", "unused"})
public class Mention
        extends AbstractNode<Mention>
        implements CaptionContent, InlineContent {

    private static final List<String> MENTION_GLOBAL_IDS = List.of("all", "here");
    private static final String MENTION_UNKNOWN = "@unknown";

    static Factory<Mention> FACTORY = new Factory<>(Type.MENTION, Mention.class, Mention::parse);

    private String id;

    @Nullable
    private String accessLevel;  // Should probably use the enum, but see docs on that for why we don't.
    @Nullable
    private String text;
    @Nullable
    private UserType userType;

    private Mention(String id) {
        this.id = validateId(id);
    }


    /**
     * Create a new mention using the given Atlassian Account ID or collection name.
     *
     * @param id the Atlassian Account ID or collection name
     * @return the new mention node
     */
    public static Mention mention(String id) {
        return new Mention(id);
    }

    public Mention id(String id) {
        this.id = validateId(id);
        return this;
    }

    /**
     * Returns the ID of the mention's target.
     *
     * @return the ID of the mention's target.
     */
    public String id() {
        return id;
    }

    //TODO The ADF documentation for this node type is incomplete. These attribute setters need better descriptions!

    /**
     * Returns the access level for this mention, if set.
     *
     * @return the access level for this mention, or {@code empty()} if not set.
     */
    public Optional<String> accessLevel() {
        return Optional.ofNullable(accessLevel);
    }

    /**
     * Sets the access level value for the mention.
     *
     * @return {@code this}
     */
    public Mention accessLevel(@Nullable String accessLevel) {
        this.accessLevel = accessLevel;
        return this;
    }

    /**
     * Sets the access level value for the mention.
     *
     * @return {@code this}
     */
    public Mention accessLevel(@Nullable AccessLevel accessLevel) {
        this.accessLevel = (accessLevel != null) ? accessLevel.name() : null;
        return this;
    }

    /**
     * Returns the user type for this mention, if set.
     *
     * @return the user type for this mention, or {@code empty()} if not set.
     */
    public Optional<UserType> userType() {
        return Optional.ofNullable(userType);
    }

    /**
     * Sets the user type value for the mention.
     *
     * @return {@code this}
     */
    public Mention userType(@Nullable String userType) {
        if (userType == null || userType.isEmpty()) {
            this.userType = null;
            return this;
        }
        return userType(UserType.PARSER.parse(userType));
    }

    /**
     * Sets the user type value for the mention.
     *
     * @return {@code this}
     */
    public Mention userType(@Nullable UserType userType) {
        this.userType = userType;
        return this;
    }

    /**
     * Returns the display text for this mention, if set.
     *
     * @return the display text for this mention, or {@code empty()} if not set.
     */
    public Optional<String> text() {
        return Optional.ofNullable(text);
    }

    /**
     * Sets the text value for the mention.
     * <br>
     * NB: As a temporary change (to replicate the behaviour of the TypeScript transformer) until the
     * {@code OMIT_EMPTY_ATTRIBUTES} parser feature flag is enabled: if {@code text} is an empty string,
     * it will be treated as {@code text} (i.e. "").
     * <br>
     * Once the feature flag is enabled, we will return to the original behaviour: if {@code text} is an empty string,
     * it will be treated as {@code null}.
     * <br>
     *  If it isn't empty but also does not begin with {@code @}, then the value will be
     * prefixed with it automatically.
     *
     * @param text the text that is displayed for the mention, which must begin with an {@code @} char.
     * @return {@code this}
     */
    public Mention text(@Nullable String text) {
        if (text == null) {
            this.text = null;
        } else if (text.isEmpty()) {
            this.text = text;
        } else if (text.charAt(0) == '@') {
            this.text = text;
        } else {
            this.text = "@" + text;
        }
        return this;
    }

    @Override
    protected boolean nodeEquals(Mention other) {
        return userType == other.userType
                && id.equals(other.id)
                && Objects.equals(accessLevel, other.accessLevel)
                && Objects.equals(text, other.text);
    }

    @Override
    protected int nodeHashCode() {
        return Objects.hash(accessLevel, id, text, userType);
    }

    @Override
    protected void appendNodeFields(ToStringHelper buf) {
        buf.appendTextField(text);
        buf.appendField("id", id);
        buf.appendField("accessLevel", accessLevel);
        buf.appendField("userType", userType);
    }

    @Override
    public Mention copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.MENTION;
    }

    @Override
    public void validate() {
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map()
                        .add(Attr.ID, id)
                        .addIfPresent(Attr.TEXT, text)
                        .addIfPresent(Attr.ACCESS_LEVEL, accessLevel)
                        .addMappedIfPresent(Attr.USER_TYPE, userType, UserType::name)
                );
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        if (MENTION_GLOBAL_IDS.contains(id)) {
            sb.append('@').append(id);
        } else {
            String text = this.text;
            sb.append((text != null) ? text : MENTION_UNKNOWN);
        }
    }

    private static Mention parse(Map<String, ?> map) {
        checkType(map, Type.MENTION);
        String id = getAttrOrThrow(map, Attr.ID);

        Mention mention = new Mention(id);
        getAttr(map, Attr.TEXT, String.class).ifPresent(mention::text);
        getAttr(map, Attr.ACCESS_LEVEL, String.class).ifPresent(mention::accessLevel);
        getAttr(map, Attr.USER_TYPE, String.class).ifPresent(mention::userType);
        return mention;
    }

    //TODO We could do extra id validation here, but apparently we need to tolerate a lot of possible
    // values, here, including an empty string.
    // The spec says "Atlassian Account ID or collection name"
    private static String validateId(String id) {
        return requireNonNull(id, "id");
    }

    /**
     * The well-known {@code accessLevel} enumerated values that were originally documented for mention nodes.
     * <p>
     * Nothing in the schema has ever restricted these values, and it seems to be common to run across
     * an empty string for this value, so these well-known values are currently only used on the value
     * setting side; the string setter (and corresponding getter) do not use this enumeration.
     *
     * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/mention/">Node - mention</a>
     */
    public enum AccessLevel {
        NONE,
        SITE,
        APPLICATION,
        CONTAINER
    }

    public enum UserType {
        DEFAULT,
        SPECIAL,
        APP;

        static EnumParser<UserType> PARSER = new EnumParser<>(UserType.class, UserType::name);
    }
}
