package com.atlassian.adf.model.node;

import com.atlassian.adf.model.ex.node.MediaException;
import com.atlassian.adf.model.mark.Link;
import com.atlassian.adf.model.mark.Mark;
import com.atlassian.adf.model.mark.type.MediaSingleMark;
import com.atlassian.adf.model.node.type.CaptionContent;
import com.atlassian.adf.model.node.type.ContentNode;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.LayoutColumnContent;
import com.atlassian.adf.model.node.type.ListItemContent;
import com.atlassian.adf.model.node.type.NestedExpandContent;
import com.atlassian.adf.model.node.type.NonNestableBlockContent;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;
import com.atlassian.adf.util.FieldMap;
import com.atlassian.annotations.Internal;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.atlassian.adf.model.ex.AdfException.frame;
import static com.atlassian.adf.model.node.AbstractContentNode.getRequiredContentMaps;
import static com.atlassian.adf.model.node.NodeParserSupport.getNodeOfType;
import static com.atlassian.adf.util.Cast.unsafeCast;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.*;
import static java.util.Objects.requireNonNull;

/**
 * A container for exactly <em>one</em> media item, optionally followed by a {@link Caption caption}.
 * This node enables the display of the content in full, in contrast to a {@link MediaGroup mediaGroup} that is
 * intended for a list of attachments. A common use case is to display an image, but it can also be used for
 * videos, or other types of content usually renderable by an {@code @atlaskit/media} card component.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #mediaSingle() mediaSingle}()
 *     .{@link Partial.NeedsMedia#center() center}()
 *     .{@link Partial.NeedsMedia#width(Number) width}(50.0)
 *     .{@link Partial.NeedsMedia#media(Media) media}(
 *         Media.{@link Media#fileMedia()} () fileMedia}()
 *                 .{@link Media.Partial.FileNeedsId#id(String) id}("ABC-123")
 *                 .{@link Media.FileOrLinkMedia#collection(String) collection}("some-collection-id")
 *                 .{@link Media.FileOrLinkMedia#occurrenceKey(String) occurrenceKey}("some-occurrence-key")
 *                 .{@link Media.FileOrLinkMedia#size(Number, Number) size}(640, 480)
 *     );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "mediaSingle",
 *     "content": [
 *       {
 *         "type": "media",
 *         "attrs": {
 *           "width": 640,
 *           "height": 480,
 *           "id": "ABC-123",
 *           "type": "file",
 *           "collection": "some-collection-id",
 *           "occurrenceKey": "some-occurrence-key"
 *         }
 *       }
 *     ],
 *     "attrs": {
 *       "layout": "center",
 *       "width": 50.0
 *     }
 *   }
 * }</pre>
 * <h2>Implementation Note</h2>
 * Although `mediaSingle` uses the `content` section to wrap other nodes, its behaviour is very different
 * from most other content nodes in that it can only contain exactly one or two nodes, the first node must
 * be a {@link Media media} node, and (when present) the second node must be a {@link Caption caption}.
 * This makes it really hard to do anything useful with the type system, so rather than do something
 * really awkward, this class just doesn't bother to implement {@code ContentNode} at all.
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/mediaSingle/">Node - mediaSingle</a>
 */
public class MediaSingle
        extends AbstractMarkedNode<MediaSingle, MediaSingleMark>
        implements DocContent, LayoutColumnContent, ListItemContent, NestedExpandContent, NonNestableBlockContent,
        TableCellContent {

    static Factory<MediaSingle> FACTORY = new Factory<>(Type.MEDIA_SINGLE, MediaSingle.class, MediaSingle::parse);

    private Layout layout;
    private Media media;

    private Width width = Width.UNSET;
    @Nullable
    private Caption caption;

    private MediaSingle(Layout layout, Media media) {
        this.layout = requireNonNull(layout, "layout");
        this.media = requireNonNull(media, "media");
    }

    public static Partial.NeedsMedia mediaSingle() {
        return new Partial.NeedsMedia(Layout.CENTER);
    }

    public static MediaSingle mediaSingle(Media media) {
        return new MediaSingle(Layout.CENTER, media);
    }

    /**
     * Creates a new partially constructed {@code mediaSingle} node with the given layout.
     *
     * @param layout see {@link Layout} for explanations of these values
     * @return a partially constructed {@code mediaSingle} node
     */
    public static Partial.NeedsMedia mediaSingle(Layout layout) {
        return new Partial.NeedsMedia(layout);
    }

    /**
     * Creates a new {@code mediaSingle} node with the given layout and media item.
     *
     * @param layout see {@link Layout} for explanations of these values
     * @param media  the {@code media} node that it should contain
     * @return the new {@code mediaSingle} container, with the given media item as content
     */
    public static MediaSingle mediaSingle(String layout, Media media) {
        return new MediaSingle(Layout.PARSER.parse(layout), media);
    }

    /**
     * Creates a new {@code mediaSingle} node with the given layout and media item.
     *
     * @param layout see {@link Layout} for explanations of these values
     * @param media  the {@code media} node that it should contain
     * @return the new {@code mediaSingle} container, with the given media item as content
     */
    public static MediaSingle mediaSingle(Layout layout, Media media) {
        return new MediaSingle(layout, media);
    }

    /**
     * Returns the display width for this single media item, if set.
     *
     * @return the display width for this single media item, or {@code empty()} if not set.
     */
    public Optional<Number> width() {
        return width.width();
    }

    /**
     * Returns the width type, which controls how the {@link #width() width} is interpreted.
     * This is always {@code empty} when the {@code width} is. If the width is set and this still
     * returns {@code empty}, then the {@code width} is interpreted as a {@link WidthType#PERCENTAGE percentage}.
     */
    public Optional<WidthType> widthType() {
        return width.widthType();
    }

    /**
     * Sets the width of the single media node's display.
     * This clears the {@link #widthType() widthType} property, which causes the value to be
     * interpreted as a {@link WidthType#PERCENTAGE percentage} for backward compatibility.
     *
     * @param width a value from {@code 0.0} to {@code 100.0} (inclusive)
     * @return {@code this}
     */
    public MediaSingle width(@Nullable Number width) {
        this.width = Width.width(width, null);
        return this;
    }

    /**
     * Sets the width of the single media node's display, as a {@link WidthType#PERCENTAGE percentage}.
     *
     * @param width a value from {@code 0.0} to {@code 100.0} (inclusive)
     * @return {@code this}
     */
    public MediaSingle widthAsPercentage(@Nullable Number width) {
        this.width = Width.width(width, WidthType.PERCENTAGE);
        return this;
    }

    /**
     * Sets the width of the single media node's display, as a {@link WidthType#PIXEL pixel} count.
     *
     * @param width a non-negative integer
     * @return {@code this}
     */
    public MediaSingle widthInPixels(@Nullable Number width) {
        this.width = Width.width(width, WidthType.PIXEL);
        return this;
    }

    public Media media() {
        return media;
    }

    public MediaSingle media(Media media) {
        this.media = requireNonNull(media, "media");
        return this;
    }

    /**
     * Returns the caption for the media item, if set.
     *
     * @return the caption for the media item, or {@code empty()} if not set.
     */
    public Optional<Caption> caption() {
        return Optional.ofNullable(caption);
    }

    public MediaSingle caption(@Nullable Caption caption) {
        this.caption = caption;
        return this;
    }

    public MediaSingle caption(String captionContent) {
        this.caption = Caption.caption(captionContent);
        return this;
    }

    public MediaSingle caption(String... captionContent) {
        this.caption = Caption.caption(captionContent);
        return this;
    }

    public MediaSingle caption(CaptionContent captionContent) {
        this.caption = Caption.caption(captionContent);
        return this;
    }

    public MediaSingle caption(CaptionContent... captionContent) {
        this.caption = Caption.caption(captionContent);
        return this;
    }

    public MediaSingle caption(Iterable<? extends CaptionContent> captionContent) {
        this.caption = Caption.caption(captionContent);
        return this;
    }

    public MediaSingle caption(Stream<? extends CaptionContent> captionContent) {
        this.caption = Caption.caption(captionContent);
        return this;
    }

    /**
     * Returns the {@link Layout} setting for this single media item.
     *
     * @return the {@link Layout} setting for this single media item.
     */
    public Layout layout() {
        return layout;
    }

    public MediaSingle layout(String layout) {
        this.layout = Layout.PARSER.parse(layout);
        return this;
    }

    public MediaSingle layout(Layout layout) {
        this.layout = requireNonNull(layout, "layout");
        return this;
    }

    public MediaSingle wrapLeft() {
        return layout(Layout.WRAP_LEFT);
    }

    public MediaSingle center() {
        return layout(Layout.CENTER);
    }

    public MediaSingle wrapRight() {
        return layout(Layout.WRAP_RIGHT);
    }

    public MediaSingle wide() {
        return layout(Layout.WIDE);
    }

    public MediaSingle fullWidth() {
        return layout(Layout.FULL_WIDTH);
    }

    public MediaSingle alignStart() {
        return layout(Layout.ALIGN_START);
    }

    public MediaSingle alignEnd() {
        return layout(Layout.ALIGN_END);
    }

    @Override
    public MediaSingle copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.MEDIA_SINGLE;
    }

    protected boolean markedNodeEquals(MediaSingle other) {
        return layout == other.layout
                && media.equals(other.media)
                && width.equals(other.width)
                && Objects.equals(caption, other.caption);
    }

    @Override
    protected int markedNodeHashCode() {
        return Objects.hash(layout, media, width, caption);
    }

    @Override
    protected void appendMarkedNodeFields(ToStringHelper buf) {
        buf.appendField("layout", layout);
        buf.appendField("width", width);
        buf.appendField("media", media);
        buf.appendField("caption", caption);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        media.appendPlainText(sb);
        Caption caption = this.caption;
        if (caption != null) caption.appendPlainText(sb);
    }

    @Override
    public Class<MediaSingleMark> markClass() {
        return MediaSingleMark.class;
    }

    @Override
    public MediaSingle mark(MediaSingleMark mark) {
        marks.add(mark);
        return this;
    }

    public MediaSingle link(@Nullable Link link) {
        marks.remove(Mark.Type.LINK);
        if (link != null) marks.add(link);
        return this;
    }

    public MediaSingle link(@Nullable String href) {
        return link((href != null) ? Link.link(href) : null);
    }

    public Optional<Link> link() {
        return marks.get(Mark.Type.LINK)
                .map(Link.class::cast);
    }

    @Override
    public Map<String, ?> toMap() {
        List<Map<String, ?>> content = new ArrayList<>(2);
        content.add(media.toMap());
        Optional.ofNullable(caption)
                .map(Node::toMap)
                .ifPresent(content::add);
        return mapWithType()
                .add(Key.CONTENT, content)
                .add(Key.ATTRS, map()
                        .addMapped(Attr.LAYOUT, layout, Layout::layout)
                        .let(width::addToAttrMap)
                )
                .let(marks::addToMap);
    }

    private static MediaSingle parse(Map<String, ?> map) {
        checkType(map, Type.MEDIA_SINGLE);
        Layout layout = Layout.PARSER.parse(getAttrOrThrow(map, Attr.LAYOUT, String.class));
        List<Map<String, ?>> contentMaps = getRequiredContentMaps(map);

        Media media = frame("[0]",
                () -> getNodeOfType(Media.class, contentMaps.get(0), Type.MEDIA_SINGLE));
        MediaSingle mediaSingle = new MediaSingle(layout, media);
        mediaSingle.width = Width.parseWidth(map);
        mediaSingle.parseMarks(map);

        int count = contentMaps.size();
        if (count > 1) {
            frame("[1]",
                    () -> mediaSingle.caption(getNodeOfType(Caption.class, contentMaps.get(1), mediaSingle)));
            if (count > 2) {
                throw new MediaException.TooManyContentItems(count).backtrace("[2]");
            }
        }

        return mediaSingle;
    }

    /**
     * This mimics the behavior of {@link ContentNode#allNodes()}, for convenience.
     *
     * @return a stream of this node and the deep expansion of its contents
     * @see MediaSingle Implementation Note
     */
    public Stream<Node> allNodes() {
        Caption caption = this.caption;
        Stream<Node> stream = Stream.of(this, media);
        if (caption != null) {
            stream = Stream.concat(stream, caption.allNodes());
        }
        return stream;
    }

    /**
     * This mimics the behavior of {@link ContentNode#allNodesOfType(Class)}.
     *
     * @see MediaSingle Implementation Note
     */
    public <T extends Node> Stream<T> allNodesOfType(Class<T> nodeClass) {
        return allNodes()
                .filter(nodeClass::isInstance)
                .map(nodeClass::cast);
    }

    /**
     * This mimics the behavior of {@link ContentNode#allNodesOfTypeAsList(Class)}.
     *
     * @see MediaSingle Implementation Note
     */
    public <T extends Node> List<T> allNodesOfTypeAsList(Class<T> nodeClass) {
        return allNodesOfType(nodeClass).collect(Collectors.toList());
    }

    /**
     * Internal helper method so that {@code MediaSingle} correctly honors the expected logic for
     * {@link ContentNode#transformDescendants(Class, Function)}.
     * <p>
     * It is not valid to have a {@code mediaSingle} node without its {@code media} content node. Therefore,
     * if this node's {@code media} content is an instance of {@code targetNodeClass} and the {@code transformer}
     * returns {@code null} for it, then this method immediately returns {@code true}. The caller should
     * then remove this entire {@code mediaSingle} node.
     * <p>
     * This is a significantly different contract from that of {@code ContentNode.transformDescendants}, so
     * its altered name, {@code @Internal} annotation, and package scope to deter general-purpose use.
     *
     * @param targetNodeClass as for {@code ContentNode.transformDescendants}
     * @param transformer     as for {@code ContentNode.transformDescendants}
     * @param <T>             as for {@code ContentNode.transformDescendants}
     * @return {@code true} to remove this node; {@code false} to retain it
     */
    @CheckReturnValue
    @Internal
    <T extends Node> boolean transformDescendantsInternal(
            Class<T> targetNodeClass,
            Function<? super T, ? extends T> transformer
    ) {
        if (targetNodeClass.isInstance(media)) {
            T result = transformer.apply(targetNodeClass.cast(media));
            if (result == null) return true;
            this.media = unsafeCast(result);
        }

        if (targetNodeClass.isInstance(caption)) {
            caption = unsafeCast(transformer.apply(targetNodeClass.cast(caption)));
        }

        // Caption is also a ContentNode, so we need to recurse...
        if (caption != null) caption.transformDescendants(targetNodeClass, transformer);
        return false;
    }

    /**
     * Types that represent a partially constructed {@link MediaSingle mediaSingle}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code mediaSingle} still needs its {@code media} content set.
         */
        class NeedsMedia {
            private Layout layout;

            // Despite the complications to this code, support for specifying the width first is provided
            // because it makes the consumer's code look a lot nicer to declare the width before the media
            // node and (if present) its caption.
            private Width width = Width.UNSET;

            NeedsMedia(Layout layout) {
                this.layout = requireNonNull(layout, "layout");
            }

            public NeedsMedia layout(String layout) {
                return layout(Layout.PARSER.parse(layout));
            }

            public NeedsMedia layout(Layout layout) {
                this.layout = requireNonNull(layout, "layout");
                return this;
            }

            public NeedsMedia wrapLeft() {
                return layout(Layout.WRAP_LEFT);
            }

            public NeedsMedia center() {
                return layout(Layout.CENTER);
            }

            public NeedsMedia wrapRight() {
                return layout(Layout.WRAP_RIGHT);
            }

            public NeedsMedia wide() {
                return layout(Layout.WIDE);
            }

            public NeedsMedia fullWidth() {
                return layout(Layout.FULL_WIDTH);
            }

            public NeedsMedia alignStart() {
                return layout(Layout.ALIGN_START);
            }

            public NeedsMedia alignEnd() {
                return layout(Layout.ALIGN_END);
            }

            @CheckReturnValue
            public NeedsMedia width(@Nullable Number width) {
                this.width = Width.width(width, null);
                return this;
            }

            @CheckReturnValue
            public NeedsMedia widthAsPercentage(@Nullable Number width) {
                this.width = Width.width(width, WidthType.PERCENTAGE);
                return this;
            }

            @CheckReturnValue
            public NeedsMedia widthInPixels(@Nullable Number width) {
                this.width = Width.width(width, WidthType.PIXEL);
                return this;
            }

            @CheckReturnValue
            public MediaSingle media(Media media) {
                MediaSingle mediaSingle = new MediaSingle(layout, media);
                mediaSingle.width = this.width;
                return mediaSingle;
            }
        }
    }

    /**
     * Layout that determines how the media idea should be positioned in relation to the flow of other information
     * on the page.
     */
    public enum Layout {
        /**
         * Provide an image floated to the left of the page with text wrapped around it.
         */
        WRAP_LEFT("wrap-left"),

        /**
         * Aligns the image in the center of the content as a block.
         * This will be used by default if no layout is explicitly supplied.
         */
        CENTER("center"),

        /**
         * Provide an image floated to the right of the page with text wrapped around it.
         */
        WRAP_RIGHT("wrap-right"),

        /**
         * Aligns the image in the center of the content as a block, but expands into the default margins
         * to produce a wider image.
         */
        WIDE("wide"),

        /**
         * Stretches the image the entire width of the page (or the surrounding container, such as a table cell).
         */
        FULL_WIDTH("full-width"),

        ALIGN_START("align-start"),

        ALIGN_END("align-end");

        static final EnumParser<Layout> PARSER = new EnumParser<>(Layout.class, Layout::layout);

        private final String layout;

        Layout(String layout) {
            this.layout = layout;
        }

        public String layout() {
            return layout;
        }
    }

    /**
     * Determines whether the {@code width} attribute should be interpreted as a percentage or pixel count.
     * Note that this is never set directly; it is instead set implicitly by which setter is used for the
     * width.
     */
    public enum WidthType {
        PERCENTAGE("percentage"),
        PIXEL("pixel");

        static final EnumParser<WidthType> PARSER = new EnumParser<>(WidthType.class, WidthType::widthType);

        private final String widthType;

        WidthType(String widthType) {
            this.widthType = widthType;
        }

        public String widthType() {
            return widthType;
        }
    }

    @Immutable
    static class Width {
        static final Width UNSET = new Width(null, null);

        @Nullable
        private final Number width;
        @Nullable
        private final WidthType widthType;

        private Width(@Nullable Number width, @Nullable WidthType widthType) {
            this.width = width;
            this.widthType = widthType;
        }

        static Width width(@Nullable Number width, @Nullable WidthType widthType) {
            if (width == null) return UNSET;

            if (widthType == WidthType.PIXEL) {
                return new Width(validateWidthInPixels(width), WidthType.PIXEL);
            }

            validateWidthAsPercentage(width);
            return new Width(width, widthType);
        }

        private static int validateWidthInPixels(Number width) {
            int widthInt = width.intValue();
            if (widthInt < 0) throw new MediaException.WidthMustBeValidPixelCount(widthInt);
            return widthInt;
        }

        private static void validateWidthAsPercentage(Number width) {
            double widthDouble = width.doubleValue();
            if (widthDouble < 0.0 || widthDouble > 100.0) throw new MediaException.WidthMustBeValidPercentage(width);
        }

        Optional<WidthType> widthType() {
            return Optional.ofNullable(widthType);
        }

        Optional<Number> width() {
            return Optional.ofNullable(width);
        }

        void addToAttrMap(FieldMap attrs) {
            if (width == null) return;

            attrs.add(Attr.WIDTH, width);
            attrs.addMappedIfPresent(Attr.WIDTH_TYPE, widthType, WidthType::widthType);
        }

        static Width parseWidth(Map<String, ?> map) {
            Number width = getAttr(map, Attr.WIDTH, Number.class).orElse(null);
            if (width == null) return UNSET;

            WidthType widthType = getAttr(map, Attr.WIDTH_TYPE, String.class)
                    .map(WidthType.PARSER::parse)
                    .orElse(null);

            return width(width, widthType);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (!(o instanceof Width)) return false;
            Width other = (Width) o;
            return numberEq(width, other.width) && widthType == other.widthType;
        }

        @Override
        public int hashCode() {
            return Objects.hashCode(widthType) * 31 + numberHash(width);
        }

        @Override
        public String toString() {
            return (width != null)
                    ? "Width{width=" + width + ", widthType=" + widthType + '}'
                    : "UNSET";
        }
    }
}
