package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.LayoutColumnContent;
import com.atlassian.adf.model.node.type.NestedExpandContent;
import com.atlassian.adf.model.node.type.NonNestableBlockContent;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.util.Char;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.stream.Stream;

import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * A container for several media items. Compare to {@link MediaSingle mediaSingle}, which is intended for the
 * display of a single {@link Media media} item in full.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #mediaGroup(Media[]) mediaGroup}(
 *         Media.{@link Media#media() media}()
 *                 .{@link Media.Partial.NeedsType#file file}()
 *                 .{@link Media.Partial.FileNeedsId#id(String) id}("6e7c7f2c-dd7a-499c-bceb-6f32bfbf30b5")
 *                 .{@link Media.FileOrLinkMedia#collection(String) collection}("ae730abd-a389-46a7-90eb-c03e75a45bf6"),
 *         Media.{@link Media#media() media}()
 *                 .{@link Media.Partial.NeedsType#file file}()
 *                 .{@link Media.Partial.FileNeedsId#id(String) id}("16a23024-6762-4d30-a6ec-298768507fa7")
 *                 .{@link Media.FileOrLinkMedia#collection(String) collection}("ae730abd-a389-46a7-90eb-c03e75a45bf6")
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "mediaGroup",
 *     "content": [
 *       {
 *         "type": "media",
 *         "attrs": {
 *           "type": "file",
 *           "id": "6e7c7f2c-dd7a-499c-bceb-6f32bfbf30b5",
 *           "collection": "ae730abd-a389-46a7-90eb-c03e75a45bf6"
 *         }
 *       },
 *       {
 *         "type": "media",
 *         "attrs": {
 *           "type": "file",
 *           "id": "16a23024-6762-4d30-a6ec-298768507fa7",
 *           "collection": "ae730abd-a389-46a7-90eb-c03e75a45bf6"
 *         }
 *       }
 *     ]
 *   }
 * }</pre>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/mediaGroup/">Node - mediaGroup</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
public class MediaGroup
        extends AbstractContentNode<MediaGroup, Media>
        implements DocContent, LayoutColumnContent, NestedExpandContent, NonNestableBlockContent, TableCellContent {

    private static final String MEDIA_GROUP_PREFIX = Char.PAPERCLIP + ' ';
    static Factory<MediaGroup> FACTORY = new Factory<>(Type.MEDIA_GROUP, MediaGroup.class, MediaGroup::parse);

    private MediaGroup() {
    }

    /**
     * @return a new, empty media group. At least one {@link Media media} node must be added to make it valid.
     */
    public static MediaGroup mediaGroup() {
        return new MediaGroup();
    }

    /**
     * @return a new media group with the given content
     */
    public static MediaGroup mediaGroup(Media content) {
        return mediaGroup().content(content);
    }

    /**
     * @return a new media group with the given content
     */
    public static MediaGroup mediaGroup(Media... content) {
        return mediaGroup().content(content);
    }

    /**
     * @return a new media group with the given content
     */
    public static MediaGroup mediaGroup(Iterable<? extends Media> content) {
        return mediaGroup().content(content);
    }

    /**
     * @return a new media group with the given content
     */
    public static MediaGroup mediaGroup(Stream<? extends Media> content) {
        return mediaGroup().content(content);
    }

    @Override
    public MediaGroup copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.MEDIA_GROUP;
    }

    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .let(this::addContent);
    }

    private static MediaGroup parse(Map<String, ?> map) {
        checkType(map, Type.MEDIA_GROUP);
        return mediaGroup().parseRequiredContent(map, Media.class);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        long count = content.stream()
                .filter(media -> media.file().isPresent())
                .count();
        if (count == 0L) return;

        // This is obviously not I18N-safe
        sb.append(MEDIA_GROUP_PREFIX)
                .append(count)
                .append(" File");
        if (count > 1L) sb.append('s');
    }
}
