package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.mark.type.InlineExtensionMark;
import com.atlassian.adf.model.node.type.ExtensionNode;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nullable;
import java.util.Map;
import java.util.Optional;

import static com.atlassian.adf.model.node.ExtensionSettings.extensionSettings;
import static com.atlassian.adf.model.node.unsupported.UnsupportedNode.plainTextFallback;

/**
 * Extensions provide hook points for ecosystem add-ons to integrate with how ADF content is rendered.
 * The {@code inlineExtension} node type is used in contexts that only permit "inline" content, such as
 * {@code paragraph} and {@code heading}.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class InlineExtension
        extends AbstractMarkedNode<InlineExtension, InlineExtensionMark>
        implements ExtensionNode<InlineExtension, InlineExtensionMark>, InlineContent {

    static final Factory<InlineExtension> FACTORY = new Factory<>(
            Type.INLINE_EXTENSION,
            InlineExtension.class,
            InlineExtension::parse
    );

    private final ExtensionSettings settings;

    private InlineExtension(ExtensionSettings settings) {
        this.settings = settings;
    }

    @CheckReturnValue
    public static ExtensionSettings.Partial.NeedsExtensionKey<InlineExtension> inlineExtension() {
        return new ExtensionSettings.Partial.NeedsExtensionKey<>(InlineExtension::new);
    }

    public static InlineExtension inlineExtension(String extensionKey, String extensionType) {
        return extensionSettings(InlineExtension::new)
                .extensionKey(extensionKey)
                .extensionType(extensionType);
    }

    @Override
    public InlineExtension copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.INLINE_EXTENSION;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, settings.toExtensionAttrs())
                .let(marks::addToMap);
    }

    @Override
    public Class<InlineExtensionMark> markClass() {
        return InlineExtensionMark.class;
    }

    @Override
    protected int markedNodeHashCode() {
        return settings.hashCode();
    }

    @Override
    protected boolean markedNodeEquals(InlineExtension other) {
        return settings.equals(other.settings);
    }

    @Override
    protected void appendMarkedNodeFields(ToStringHelper buf) {
        settings.appendExtensionFields(buf);
    }

    @Override
    public String extensionKey() {
        return settings.extensionKey();
    }

    @Override
    public InlineExtension extensionKey(String extensionKey) {
        settings.extensionKey(extensionKey);
        return this;
    }

    @Override
    public String extensionType() {
        return settings.extensionType();
    }

    @Override
    public InlineExtension extensionType(String extensionType) {
        settings.extensionType(extensionType);
        return this;
    }

    @Override
    public InlineExtension localId(@Nullable String localId) {
        settings.localId(localId);
        return this;
    }

    @Override
    public Optional<String> localId() {
        return settings.localId();
    }

    @Override
    public InlineExtension text(@Nullable String text) {
        settings.text(text);
        return this;
    }

    @Override
    public Optional<String> text() {
        return settings.text();
    }

    @Override
    public InlineExtension parameters(@Nullable Map<String, ?> parameters) {
        settings.parameters(parameters);
        return this;
    }

    @Override
    public Optional<Map<String, ?>> parameters() {
        return settings.parameters();
    }

    private static InlineExtension parse(Map<String, ?> map) {
        ExtensionSettings settings = ExtensionSettings.parse(map);
        return new InlineExtension(settings).parseMarks(map);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        String text = settings.text().orElseGet(() -> plainTextFallback(this));
        sb.append(text);
    }
}
